<?php

declare(strict_types=1);

namespace Drupal\Tests\entity_usage_plus\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\field\Traits\EntityReferenceFieldCreationTrait;

/**
 * Test overwritten controller from entity_usage.
 *
 * @group entity_usage_plus
 */
class EntityUsagePlusControllerTest extends BrowserTestBase {
  use EntityReferenceFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'entity_usage_plus',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * User with admin privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * Target content type.
   *
   * @var string
   */
  protected $targetType = 'target_type';

  /**
   * Source content type.
   *
   * @var string
   */
  protected $sourceType = 'source_type';

  /**
   * Entity node reference field on source type.
   *
   * @var string
   */
  protected $sourceField = 'field_node_reference';

  /**
   * Title of target node.
   *
   * @var string
   */
  protected $targetTitle = 'Target node title';

  /**
   * Title of source node.
   *
   * @var string
   */
  protected $sourceTitle = 'Source node title';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Configure entity_usage to track node reference fields.
    $this->config('entity_usage.settings')
      ->set('local_task_enabled_entity_types', ['node'])
      ->set('track_enabled_source_entity_types', ['node'])
      ->set('track_enabled_target_entity_types', ['node'])
      ->set('track_enabled_plugins', ['entity_reference'])
      ->save();

    // Clear routing cache.
    \Drupal::service('router.builder')->rebuild();

    // Create 2 node types for the target and source entities.
    $this->createContentType([
      'type' => $this->targetType,
      'name' => 'Target Bundle',
    ]);
    $this->createContentType([
      'type' => $this->sourceType,
      'name' => 'Source Bundle',
    ]);
    // Add a node reference field on source type.
    $this->createEntityReferenceField(
      'node',
      $this->sourceType,
      $this->sourceField,
      'Node Ref',
      'node',
      'default',
      ['target_bundles' => [$this->targetType => $this->targetType]],
    );

    // Create target node 1.
    $this->createNode([
      'type' => $this->targetType,
      'title' => $this->targetTitle,
    ]);
    $target_node = $this->drupalGetNodeByTitle($this->targetTitle);
    $this->assertNotEmpty($target_node, 'Target node found in database.');

    // Create source node 2 referencing 1.
    $this->createNode([
      'type' => $this->sourceType,
      'title' => $this->sourceTitle,
      $this->sourceField => [
        'target_id' => $target_node->id(),
      ],
    ]);

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'access entity usage statistics',
      'administer entity usage',
    ]);

    // Login.
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Test default controller.
   */
  public function testDefaultController(): void {
    // View usage of first node.
    $target_node = $this->drupalGetNodeByTitle($this->targetTitle);
    $this->assertNotEmpty($target_node, 'Target node found in database.');
    $this->drupalGet('node/' . $target_node->id() . '/usage');
    $this->assertSession()->pageTextContains($this->sourceTitle);

    // View usage of second node.
    $source_node = $this->drupalGetNodeByTitle($this->sourceTitle);
    $this->assertNotEmpty($source_node, 'Source node found in database.');
    $this->drupalGet('node/' . $source_node->id() . '/usage');
    $this->assertSession()->pageTextContains('There are no recorded usages for entity of type: node with id: 2');
  }

  /**
   * Test overwritten controller.
   */
  public function testOverwrittenController(): void {
    // Update entity_usage_plus to override controller.
    $this->drupalGet('admin/config/entity-usage/settings/entity-usage-plus');

    // Assure we loaded the settings form.
    $this->assertSession()->statusCodeEquals(200);

    // Update the settings.
    $this->submitForm(
      ['override_tab_display' => TRUE],
      'Save configuration',
    );

    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Clear routing cache.
    \Drupal::service('router.builder')->rebuild();

    // View usage of first node.
    $target_node = $this->drupalGetNodeByTitle($this->targetTitle);
    $this->assertNotEmpty($target_node, 'Target node found in database.');
    $this->drupalGet('node/' . $target_node->id() . '/usage');
    $this->assertSession()->pageTextContains($this->sourceTitle);

    // View usage of second node.
    // Child entities should be listed.
    $source_node = $this->drupalGetNodeByTitle($this->sourceTitle);
    $this->assertNotEmpty($source_node, 'Source node found in database.');
    $this->drupalGet('node/' . $source_node->id() . '/usage');
    $this->assertSession()->pageTextContains($this->targetTitle);
  }

}
