<?php

namespace Drupal\entity_uuid_lookup\Form;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an admin form to lookup entities by UUID.
 */
class EntityUuidLookupForm extends FormBase {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity repository.
   */
  protected EntityRepositoryInterface $entityRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityRepository = $container->get('entity.repository');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'entity_uuid_lookup_admin';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['uuid'] = [
      '#type' => 'textfield',
      '#title' => $this->t('UUID'),
      '#description' => $this->t('The UUID of the entity to redirect to.'),
      '#required' => TRUE,
    ];

    $form['view'] = [
      '#type' => 'submit',
      '#value' => $this->t('View'),
    ];

    $form['edit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Edit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $uuid = $form_state->getValue('uuid');
    $triggeringElement = $form_state->getTriggeringElement();
    $entityTypeIds = $this->entityTypeManager->getDefinitions();

    foreach ($entityTypeIds as $entityTypeId => $entityType) {
      if (!$entityType->getKey('uuid')) {
        continue;
      }

      $entity = $this->entityRepository->loadEntityByUuid($entityTypeId, $uuid);
      if (!$entity instanceof EntityInterface) {
        continue;
      }

      if ($entity->hasLinkTemplate('edit-form') && $triggeringElement['#id'] === 'edit-edit') {
        $editUrl = $entity->toUrl('edit-form');

        if ($editUrl->access()) {
          $url = $editUrl;
        }
      }

      if (!isset($url) && $entity->hasLinkTemplate('canonical')) {
        $url = $entity->toUrl('canonical');
      }

      if (!isset($url) || !$url->access()) {
        break;
      }

      $form_state->setRedirectUrl($url);
      return;
    }

    $form_state->setErrorByName('uuid', $this->t('No entity found with the given UUID.'));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
  }

}
