<?php

namespace Drupal\entitygroupfield_lite\Field;

use Drupal\Core\Field\EntityReferenceFieldItemList;
use Drupal\Core\TypedData\ComputedItemListTrait;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\group\Entity\GroupRelationship;
use Drupal\group\Entity\GroupInterface;

/**
 * A computed property for the related groups.
 */
class EntityGroupFieldItemList extends EntityReferenceFieldItemList {

  use ComputedItemListTrait;

  /**
   * {@inheritdoc}
   */
  public function computeValue() {
    // No value will exist if the entity has not been created, so exit early.
    if ($this->getEntity()->isNew()) {
      return NULL;
    }

    // If this entity/bundle has no group relation type plugins enabled,
    // there's no way there could be any group associations, so exit early.
    if (!$this->getGroupRelationTypePluginIds($this->getEntity()->getEntityTypeId(), $this->getEntity()->bundle())) {
      return NULL;
    }

    $group_relationships = GroupRelationship::loadByEntity($this->getEntity());
    if (empty($group_relationships)) {
      return NULL;
    }

    $group_type_id = $this->getFieldDefinition()->getSettings()['group_type_id'];
    $this->list = [];
    foreach ($group_relationships as $delta => $group_relationship) {
      if ($group_type_id == $group_relationship->getGroupTypeId()) {
        $this->list[] = $this->createItem($delta, [
          'target_id' => $group_relationship->getGroupId(),
        ]);
      }
    }
  }

  /**
   * {@inheritdoc}
   *
   * We need to override the presave to avoid saving without the host entity id
   * generated.
   */
  public function preSave() {
  }

  /**
   * {@inheritdoc}
   */
  public function postSave($update) {
    if ($this->valueComputed) {
      $host_entity = $this->getEntity();
      $relationships_existing = $this->getExistingRelationships($host_entity);
      $relations_new = [];
      // New or existing relationships.
      foreach ($this->getIterator() as $item) {
        $group = $item->entity;

        $existing = $this->getExistingRelationship($group, $relationships_existing);
        if (empty($existing)) {
          $plugin_id = $this->getGroupRelationshipPluginId($host_entity->getEntityTypeId(), $host_entity->bundle());
          $relationship = $group->addRelationship($host_entity, $plugin_id);
          $relations_new[$relationship->id()] = $relationship;
        }
        else {
          $relations_new[$existing->id()] = $existing;
        }
      }

      // Deleted relationships.
      $diff = array_diff_key($relationships_existing, $relations_new);
      if (!empty($diff)) {
        \Drupal::entityTypeManager()->getStorage('group_relationship')->delete($diff);
      }
    }
    return parent::postSave($update);
  }

  /**
   * Returns the relation type plugin IDs for a given entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The entity bundle.
   *
   * @return mixed
   *   The plugin id or false.
   */
  protected function getGroupRelationshipPluginId(string $entity_type_id, string $bundle) {
    foreach (\Drupal::service('group_relation_type.manager')->getDefinitions() as $plugin_id => $plugin_info) {
      if ($plugin_info->getEntityTypeId() === $entity_type_id) {
        if (isset($bundle) && !empty($plugin_info->getEntityBundle()) && $plugin_info->getEntityBundle() !== $bundle) {
          continue;
        }
        return $plugin_id;
      }
    }
    return FALSE;
  }

  /**
   * Return the first encountered relationship to the given group.
   *
   * @param \Drupal\group\Entity\GroupInterface $group
   *   The group entity.
   * @param array $relationships
   *   An array of relationships objects.
   *
   * @return mixed
   *   The relationship object or false.
   */
  protected function getExistingRelationship(GroupInterface $group, array $relationships) {
    foreach ($relationships as $relationship) {
      if ($group->id() == $relationship->getGroupId()) {
        return $relationship;
      }
    }
    return FALSE;
  }

  /**
   * Get existing relationships for the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The content entity to retrieve relationships for.
   *
   * @return array
   *   An array of group relation type plugin IDs that match.
   */
  protected function getExistingRelationships(ContentEntityInterface $entity) {
    $field_definition = $this->getFieldDefinition();
    $settings = $field_definition->getItemDefinition()->getSettings();
    $bundles = $settings['handler_settings']['target_bundles'];
    $relationships = \Drupal::entityTypeManager()->getStorage('group_relationship')->loadByEntity($entity);
    foreach ($relationships as $k => $relationship) {
      if (!array_key_exists($relationship->getGroup()->bundle(), $bundles)) {
        unset($relationships[$k]);
      }
    }
    return $relationships;
  }

  /**
   * Returns the group relation type plugin IDs for a given entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   (optional) The entity bundle.
   *
   * @return array
   *   An array of group relation type plugin IDs that match.
   */
  protected function getGroupRelationTypePluginIds(string $entity_type_id, ?string $bundle = NULL) {
    $plugin_ids = [];
    foreach (\Drupal::service('group_relation_type.manager')->getDefinitions() as $plugin_id => $plugin_info) {
      if ($plugin_info->getEntityTypeId() === $entity_type_id) {
        if (isset($bundle) && !empty($plugin_info->getEntityBundle()) && $plugin_info->getEntityBundle() !== $bundle) {
          continue;
        }
        $plugin_ids[] = $plugin_id;
      }
    }
    return $plugin_ids;
  }

}
