<?php

namespace Drupal\entitygroupfield_lite\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;

/**
 * Hook implementations.
 */
class EntityHooks {

  use StringTranslationTrait;

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Implements hook_entity_bundle_field_info().
   *
   * Attach computed fields to groups.
   */
  #[Hook('entity_bundle_field_info')]
  public function attachGroupFields(EntityTypeInterface $entity_type, $bundle, array $base_field_definitions) {
    $fields = [];
    foreach ($this->entityTypeManager->getStorage('group_type')->loadMultiple() as $group_type) {
      /** @var \Drupal\group\Entity\GroupType $group_type */
      $field_name = 'group_' . $group_type->id();

      foreach ($group_type->getInstalledPlugins() as $plugin) {
        if ($plugin->getPluginDefinition()->getEntityBundle() === FALSE) {
          if (($entity_type->id() != $plugin->getPluginDefinition()->getEntityTypeId())) {
            continue;
          }
        }
        elseif (($entity_type->id() != $plugin->getPluginDefinition()->getEntityTypeId()) || ($bundle != $plugin->getPluginDefinition()->getEntityBundle())) {
          continue;
        }

        $field_label = $this->t('Attach to the %group group', ['%group' => $group_type->label()]);
        $cardinality = ($plugin->getGroupCardinality() != 0) ? $plugin->getGroupCardinality() : FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED;
        $fields[$field_name] = BaseFieldDefinition::create('entity_reference')
          ->setName($field_name)
          ->setTargetEntityTypeId('group')
          ->setLabel($field_label)
          ->setTranslatable(FALSE)
          ->setComputed(TRUE)
          ->setCustomStorage(TRUE)
          ->setReadOnly(TRUE)
          ->setCardinality($cardinality)
          ->setSettings([
            'target_type' => 'group',
            'handler' => 'default:group',
            'group_type_id' => $group_type->id(),
            'handler_settings' => [
              'target_bundles' => [$group_type->id() => $group_type->id()],
              'sort' => ['field' => 'label', 'direction' => 'asc'],
              'auto_create' => FALSE,
              'auto_create_bundle' => '',
            ],
          ])
          ->setClass('\Drupal\entitygroupfield_lite\Field\EntityGroupFieldItemList')
          ->setDisplayConfigurable('form', TRUE)
          ->setDisplayOptions('form', [
            'region' => 'hidden',
            'weight' => 50,
          ])
          ->setDisplayConfigurable('view', TRUE)
          ->setDisplayOptions('view', [
            'label' => 'hidden',
            'region' => 'hidden',
            'weight' => 50,
          ]
        );
      }
    }
    return $fields;
  }

}
