<?php

namespace Drupal\entitygroupfield_lite\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsSelectWidget;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\group\Entity\GroupInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\group\Access\GroupPermissionCheckerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface;

/**
 * Plugin implementation of the 'options_select' widget.
 */
#[FieldWidget(
  id: 'group_select',
  label: new TranslatableMarkup('Group select list'),
  field_types: [
    'entity_reference',
  ],
  multiple_values: TRUE,
)]
class GroupSelectWidget extends OptionsSelectWidget implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $account;

  /**
   * The Group relation type manager.
   *
   * @var \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManager
   */
  protected $groupRelationTypeManager;

  /**
   * The group permission access checker service.
   *
   * @var \Drupal\group\Access\GroupPermissionCheckerInterface
   */
  protected $groupPermissionChecker;

  /**
   * Constructs a InlineEntityFormComplex object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user.
   * @param \Drupal\group\Plugin\Group\Relation\GroupRelationTypeManagerInterface $group_relation_type_manager
   *   The Group relation type manager.
   * @param \Drupal\group\Access\GroupPermissionCheckerInterface $group_permission_checker
   *   The group permission checker.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $account,
    GroupRelationTypeManagerInterface $group_relation_type_manager,
    GroupPermissionCheckerInterface $group_permission_checker,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->entityTypeManager = $entity_type_manager;
    $this->account = $account;
    $this->groupRelationTypeManager = $group_relation_type_manager;
    $this->groupPermissionChecker = $group_permission_checker;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('group_relation_type.manager'),
      $container->get('group_permission.checker')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'override_label' => '',
      'required' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element['override_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Set label'),
      '#default_value' => $this->getSetting('override_label'),
      '#description' => $this->t('Override the default label.'),
    ];
    $element['required'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Required'),
      '#default_value' => $this->getSetting('required'),
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $override_label = $this->getSetting('override_label');
    if (!empty($override_label)) {
      $summary[] = $this->t('Use field label: @override_label', ['@override_label' => $override_label]);
    }
    $summary[] = ($this->getSetting('required')) ? $this->t('Required: Yes') : $this->t('Required: No');

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    // If we are in a group context hide the form.
    if ($form_state->get('group_wizard_id')) {
      $current_group = $form_state->get('group');
      $field_settings = $this->fieldDefinition->getSettings();
      if (array_key_exists($current_group->bundle(), $field_settings['handler_settings']['target_bundles'])) {
        return [];
      }
    }

    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Set form element label.
    $override_label = $this->getSetting('override_label');
    if (!empty($override_label)) {
      $element['#title'] = $override_label;
    }

    // Set required behavior.
    if ($this->getSetting('required')) {
      $element['#required'] = TRUE;
    }

    // Adjust cardinality.
    $options = $this->options;
    unset($options['_none']);
    if (count($options) < 2) {
      $element['#multiple'] = FALSE;
    }

    // Disable options that are not allowed.
    $groups = $this->entityTypeManager->getStorage('group')->loadMultiple(array_keys($options));
    $plugin_id = $this->getRelationshipPluginId($items->getEntity()->getEntityTypeId(), $items->getEntity()->bundle());
    $enabled_options = $options;
    foreach ($groups as $gid => $group) {
      assert($group instanceof GroupInterface);
      $op = !in_array($gid, $element['#default_value']) ? 'create' : 'delete';
      if ($op == 'delete') {
        $op .= ($group->getOwnerId() == $this->account->id()) ? ' own' : ' any';
      }
      $permission = "{$op} {$plugin_id} relationship";

      if (!$this->groupPermissionChecker->hasPermissionInGroup($permission, $this->account, $group)) {
        $element['#options_attributes'][$gid]['disabled'] = 'disabled';
        unset($enabled_options[$gid]);
      }
    }

    // Transform to a single checkbox if only one option is available.
    if (count($options) == 1) {
      $element['#type'] = 'checkbox';
      $element['#default_value'] = !empty($element['#default_value']);
      $element['#return_value'] = key($options);
      unset($element['#options']);
    }

    // If all options are disabled deny access to the hole widget.
    if (empty($enabled_options)) {
      $element['#access'] = FALSE;
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $values = array_filter($values, function ($value) {
      return !empty($value['target_id']);
    });
    return $values;
  }

  /**
   * Get the relationship plugin id for the given entity type + bundle.
   *
   * @param string $entity_type_id
   *   The entity type id.
   * @param string $bundle
   *   The entity bundle.
   *
   * @return string
   *   The relationship plugin id.
   */
  protected function getRelationshipPluginId($entity_type_id, $bundle) {
    foreach ($this->groupRelationTypeManager->getDefinitions() as $plugin_id => $plugin_info) {
      if ($plugin_info->getEntityTypeId() === $entity_type_id) {
        if (!empty($plugin_info->getEntityBundle()) && $plugin_info->getEntityBundle() === $bundle) {
          return $plugin_id;
        }
      }
    }
    return '';
  }

}
