<?php

namespace Drupal\entityqueue_buttons\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure EntityQueue Buttons settings.
 */
class EntityQueueButtonsSettingsForm extends ConfigFormBase {
  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a EntityQueueButtonsSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('entity_type.manager')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'entityqueue_buttons_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['entityqueue_buttons.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('entityqueue_buttons.settings');

    // Get all content types.
    $content_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();

    // Get all entity queues.
    $entity_queues = $this->entityTypeManager->getStorage('entity_queue')->loadMultiple();

    // Build options array for entity queues.
    $queue_options = [];
    foreach ($entity_queues as $queue) {
      $queue_options[$queue->id()] = $queue->label();
    }

    $form['queue_settings'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Content Type'),
        $this->t('Enabled Queues'),
      ],
      '#empty' => $this->t('No content types available.'),
    ];

    foreach ($content_types as $content_type) {
      $content_type_id = $content_type->id();
      $form['queue_settings'][$content_type_id] = [
        'content_type' => [
          '#markup' => $content_type->label(),
        ],
        'queues' => [
          '#type' => 'checkboxes',
          '#options' => $queue_options,
          '#default_value' => $config->get("queue_settings.$content_type_id") ?: [],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('entityqueue_buttons.settings');
    $values = $form_state->getValue('queue_settings');

    foreach ($values as $content_type_id => $data) {
      if (isset($data['queues']) && is_array($data['queues'])) {
        $selected_queues = array_filter($data['queues']);
        $config->set("queue_settings.$content_type_id", $selected_queues);
      }
    }

    $config->save();
    parent::submitForm($form, $form_state);
  }

}
