<?php

namespace Drupal\enum_field\Drush;

use Consolidation\AnnotatedCommand\AnnotationData;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush hooks for integrating the enum_field module with other commands.
 */
class EnumFieldHooks extends DrushCommands {

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
  ) {
  }

  /**
   * Instantiates a new instance of this class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('module_handler'),
    );
  }

  /**
   * Add enum_field related options to a command.
   */
  #[CLI\Hook(type: HookManager::OPTION_HOOK, target: 'field:create')]
  public function hookOption(Command $command, AnnotationData $annotationData): void {
    if (!$this->isInstalled()) {
      return;
    }

    $command->addOption(
      'enum-class',
      '',
      InputOption::VALUE_OPTIONAL,
      'The enum class to use for this field.'
    );
  }

  /**
   * Handle field storage creation for enum fields.
   */
  #[CLI\Hook(type: HookManager::ON_EVENT, target: 'field-create-field-storage')]
  public function hookFieldStorage(array $values, InputInterface $input): array {
    if (in_array($input->getOption('field-type'), ['enum_integer', 'enum_string'])) {
      $values['settings']['enum_class'] = $this->getEnumClass($input);
    }

    return $values;
  }

  /**
   * Get the enum class from input or by prompting the user.
   */
  protected function getEnumClass(InputInterface $input): string {
    $value = $input->getOption('enum-class');

    if ($value === NULL && $input->isInteractive()) {
      $value = $this->askEnumClass();
    }

    $input->setOption('enum-class', $value);

    return $value;
  }

  /**
   * Prompt the user for the enum class.
   */
  protected function askEnumClass(): ?string {
    return $this->io()->ask('Enum class');
  }

  /**
   * Check whether the enum_field module is installed.
   */
  protected function isInstalled(): bool {
    return $this->moduleHandler->moduleExists('enum_field');
  }

}
