<?php

namespace Drupal\environment_indicator\Entity;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Defines an Environment configuration entity.
 *
 * @ConfigEntityType(
 *   id = "environment_indicator",
 *   label = @Translation("Environment Switcher"),
 *   handlers = {
 *     "storage" = "Drupal\Core\Config\Entity\ConfigEntityStorage",
 *     "access" = "Drupal\environment_indicator\EnvironmentIndicatorAccessControlHandler",
 *     "list_builder" = "Drupal\environment_indicator\EnvironmentIndicatorListBuilder",
 *     "form" = {
 *       "add" = "Drupal\environment_indicator\EnvironmentIndicatorForm",
 *       "edit" = "Drupal\environment_indicator\EnvironmentIndicatorForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     }
 *   },
 *   admin_permission = "administer environment indicator settings",
 *   config_prefix = "switcher",
 *   static_cache = TRUE,
 *   entity_keys = {
 *     "id" = "machine",
 *     "label" = "name",
 *     "weight" = "weight"
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/development/environment-indicator/manage/{environment_indicator}",
 *     "delete-form" = "/admin/config/development/environment-indicator/manage/{environment_indicator}/delete",
 *     "collection" = "/admin/config/development/environment-indicator"
 *   },
 *   config_export = {
 *     "machine",
 *     "name",
 *     "url",
 *     "weight",
 *     "fg_color",
 *     "bg_color",
 *   }
 * )
 */
class EnvironmentIndicator extends ConfigEntityBase implements ConfigEntityInterface {

  /**
   * The machine-readable ID for the configurable.
   *
   * @var string
   */
  protected $machine;

  /**
   * The human-readable name for the configurable.
   *
   * @var string
   */
  protected $name;

  /**
   * The URL to switch to.
   *
   * @var string
   */
  protected $url;

  /**
   * The switcher's link weight.
   *
   * @var int
   */
  protected $weight;

  /**
   * The color code for the foreground indicator.
   *
   * @var string
   */
  protected $fg_color;

  /**
   * The color code for the background indicator.
   *
   * @var string
   */
  protected $bg_color;

  /**
   * The class name of this indicator, generated from the machine name.
   *
   * @var string
   */
  protected $class_name;

  /**
   * {@inheritdoc}
   */
  public function id() {
    return $this->get('machine');
  }

  /**
   * {@inheritdoc}
   */
  public function label($langcode = NULL) {
    return $this->get('name');
  }

  /**
   * Gets the URL.
   *
   * @return string
   *   The URL to switch to.
   */
  public function getUrl() {
    return $this->get('url');
  }

  /**
   * Gets the weight.
   *
   * @return int
   *   The weight of the switcher environment.
   */
  public function getWeight() {
    return $this->get('weight');
  }

  /**
   * Gets the foreground color.
   *
   * @return string
   *   The color code for the indicator.
   */
  public function getFgColor() {
    return $this->get('fg_color');
  }

  /**
   * Gets the background color.
   *
   * @return string
   *   The color code for the indicator.
   */
  public function getBgColor() {
    return $this->get('bg_color');
  }

  /**
   * Outputs the CSS style for the environment indicator as a string.
   *
   * @param string $selector
   *   The CSS selector for the link to apply the style to.
   *
   * @return string
   *   The CSS style.
   */
  public function getLinkStyle($selector): string {
    return sprintf(
      '%s a.%s,
      %s a.%s:hover,
      %s a.%s:focus,
      %s a.%s:active,
      %s a.%s:visited {
        --environment-indicator-background-color: %s;
        --environment-indicator-color: %s;
        background-color: var(--environment-indicator-background-color);
        color: var(--environment-indicator-color);
      }',
      $selector ?? '',
      $this->getClassName() ?? '',
      $selector ?? '',
      $this->getClassName() ?? '',
      $selector ?? '',
      $this->getClassName() ?? '',
      $selector ?? '',
      $this->getClassName() ?? '',
      $selector ?? '',
      $this->getClassName() ?? '',
      $this->getBgColor() ?? '#0f0f0f',
      $this->getFgColor() ?? '#ddd'
    );
  }

  /**
   * Gets the class name.
   *
   * @return string
   *   The class name of this indicator.
   */
  public function getClassName(): string {
    // If class_name is not explicitly set, generate it from the machine name.
    if (empty($this->class_name)) {
      $this->class_name = Html::getClass('environment-indicator-switcher--' . $this->id());
    }
    return $this->class_name;
  }

  /**
   * Sets the machine name and generates the corresponding class name.
   *
   * @param string $machine
   *   The machine-readable ID for the configurable.
   */
  public function setMachine($machine) {
    $this->set('machine', $machine);
    // Automatically set the class name based on the machine name.
    $this->setClassName(Html::getClass('environment-indicator-' . $machine));
  }

  /**
   * Sets the name.
   *
   * @param string $name
   *   The environment name.
   */
  public function setName($name) {
    $this->set('name', $name);
  }

  /**
   * Sets the weight.
   *
   * @param int $weight
   *   The environment link weight.
   */
  public function setWeight($weight) {
    $this->set('weight', $weight);
  }

  /**
   * Sets the URL.
   *
   * @param string $url
   *   The environment URL.
   */
  public function setUrl($url) {
    $this->set('url', $url);
  }

  /**
   * Sets the foreground color.
   *
   * @param string $fg_color
   *   The foreground color.
   */
  public function setFgColor($fg_color) {
    $this->set('fg_color', $fg_color);
  }

  /**
   * Sets the background color.
   *
   * @param string $bg_color
   *   The background color.
   */
  public function setBgColor($bg_color) {
    $this->set('bg_color', $bg_color);
  }

  /**
   * Sets the class name.
   *
   * @param string $class_name
   *   The class name of this indicator.
   */
  protected function setClassName($class_name) {
    $this->set('class_name', $class_name);
  }

}
