<?php

declare(strict_types=1);

namespace Drupal\Tests\environment_indicator_toolbar\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests for Environment Indicator.
 *
 * @group environment_indicator
 */
class EnvironmentIndicatorGinTest extends BrowserTestBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['environment_indicator_toolbar', 'gin_toolbar'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * The path to the environment_indicator module.
   *
   * @var string
   */
  protected string $environmentIndicatorModulePath;

  /**
   * The path to the environment_indicator module.
   *
   * @var string
   */
  protected string $environmentIndicatorToolbarModulePath;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // Retrieve the dynamic module path.
    $moduleHandler = \Drupal::service('extension.list.module');
    $this->environmentIndicatorModulePath = $moduleHandler->getPath('environment_indicator');
    $this->environmentIndicatorToolbarModulePath = $moduleHandler->getPath('environment_indicator_toolbar');

    // Disable CSS preprocessing.
    $config = $this->config('system.performance');
    $config->set('css.preprocess', FALSE)->save();
    $this->assertTrue(\Drupal::service('theme_installer')->install(['gin']));
    $this->container->get('config.factory')
      ->getEditable('system.theme')
      ->set('admin', 'gin')
      ->save();

    // Create users.
    $this->privilegedUser = $this->drupalCreateUser([
      'access environment indicator',
      'access toolbar',
    ]);
    $this->unprivilegedUser = $this->drupalCreateUser();
  }

  /**
   * Tests that the gin classic horizontal toolbar integration works.
   */
  public function testEnvironmentIndicatorGinClassicHorizontal(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Gin Classic Horizontal Environment')
      ->set('fg_color', '#87ff00')
      ->set('bg_color', '#ff0000')
      ->save();
    $gin_config = $this->config('gin.settings');
    $gin_config->set('classic_toolbar', 'horizontal')
      ->save();
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar.css']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar-horizontal.css']");
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar-vertical.css']");
    $session->pageTextContains('Gin Classic Horizontal Environment');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
  }

  /**
   * Tests that the gin classic vertical toolbar integration works.
   */
  public function testEnvironmentIndicatorGinClassicVertical(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Gin Classic Vertical Environment')
      ->set('fg_color', '#87ff00')
      ->set('bg_color', '#ff0000')
      ->save();
    $gin_config = $this->config('gin.settings');
    $gin_config->set('classic_toolbar', 'vertical')
      ->save();
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementExists("css", "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar-vertical.css']");
    $session->elementExists("css", "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar.css']");
    $session->elementExists("css", "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $session->elementNotExists("css", "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar-horizontal.css']");
    $session->pageTextContains('Gin Classic Vertical Environment');

  }

  /**
   * Tests that a user without permission does not see environment indicators.
   */
  public function testGinIntegrationWithoutPermission(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'No Access Gin')
      ->set('fg_color', '#123456')
      ->set('bg_color', '#abcdef')
      ->save();

    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);

    $this->drupalLogin($this->unprivilegedUser);
    $this->drupalGet('<front>');

    $session = $this->assertSession();

    // The name should not be visible.
    $session->pageTextNotContains('No Access Gin');

    // None of the Gin-related environment indicator styles should be loaded.
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar.css']");
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar-horizontal.css']");
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar-vertical.css']");

    $page = $this->getSession()->getPage()->getContent();
    $this->assertStringNotContainsString('environmentIndicator', $page);
    $this->assertStringNotContainsString('--environment-indicator-color', $page);
  }

  /**
   * Tests that a user with permission sees environment indicators.
   */
  public function testGinIntegrationWithPermission(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'With Access Gin')
      ->set('fg_color', '#654321')
      ->set('bg_color', '#fedcba')
      ->save();

    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);

    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');

    $session = $this->assertSession();

    // The name should be visible.
    $session->pageTextContains('With Access Gin');

    // The base toolbar and gin toolbar styles should be loaded.
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/gin-toolbar.css']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");

    $page = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('environmentIndicator', $page);
    $this->assertStringContainsString('--environment-indicator-color: #654321', $page);
    $this->assertStringContainsString('--environment-indicator-background-color: #fedcba', $page);
  }

}
