<?php

declare(strict_types=1);

namespace Drupal\Tests\environment_indicator_toolbar\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the toolbar integration.
 *
 * @group environment_indicator
 */
class ToolbarIntegrationTest extends BrowserTestBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'environment_indicator_toolbar',
    'environment_indicator_ui',
    'toolbar',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to see the environment indicator.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * A user with permission to see the current release.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $currentReleaseUser;

  /**
   * The path to the environment_indicator module.
   *
   * @var string
   */
  protected string $environmentIndicatorModulePath;

  /**
   * The path to the environment_indicator_toolbar module.
   *
   * @var string
   */
  protected string $environmentIndicatorToolbarModulePath;


  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // Retrieve the dynamic module path.
    $moduleHandler = \Drupal::service('extension.list.module');
    $this->environmentIndicatorModulePath = $moduleHandler->getPath('environment_indicator');
    $this->environmentIndicatorToolbarModulePath = $moduleHandler->getPath('environment_indicator_toolbar');
    $this->state = $this->container->get('state');
    // Disable CSS preprocessing.
    $config = $this->config('system.performance');
    $config->set('css.preprocess', FALSE)->save();

    // Create users.
    $this->privilegedUser = $this->drupalCreateUser(['access environment indicator', 'access toolbar']);
    $this->unprivilegedUser = $this->drupalCreateUser();
    $this->currentReleaseUser = $this->drupalCreateUser([
      'access toolbar',
      'access environment indicator',
      'view environment indicator current release',
    ]);
  }

  /**
   * Tests that the element appears in the page top region.
   *
   * It shouldn't matter if the toolbar module is enabled,
   * and toolbar integration is disabled, if the environment
   * indicator toolbar module is enabled, the environment indicator
   * should never appear in the page top region.
   *
   * This also tests that the correct libraries are loaded.
   */
  public function testEnvironmentIndicatorVisibilityWithToolBarSettingDisabled(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Red Green Environment')
      ->set('fg_color', 'green')
      ->set('bg_color', 'red')
      ->save();
    $settings = $this->config('environment_indicator.settings');
    $settings->set('toolbar_integration', ['toolbar' => 0])
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->pageTextContains('Red Green Environment');
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/environment_indicator.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
  }

  /**
   * Tests that the element does not appear in the page top region.
   *
   * If the toolbar module is enabled and the toolbar integration is enabled,
   * the environment indicator should not appear in the page top region.
   *
   * This also tests that the correct libraries are loaded.
   */
  public function testEnvironmentIndicatorVisibilityWithToolBarSettingEnabled(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Test Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/environment_indicator.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
  }

  /**
   * Tests that CSS selectors that environment indicator uses exist.
   */
  public function testEnvironmentIndicatorToolbarIntegration(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Test Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalLogin($this->privilegedUser);
    $this->drupalGet('<front>');
    $session = $this->assertSession();
    $session->pageTextContains('Test Environment');
    $session->elementNotExists('css', '#environment-indicator');
    // Change configuration values.
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Development Environment')
      ->set('fg_color', '#efefef')
      ->set('bg_color', '#12285f')
      ->save();
    // Clear drupal cache.
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalGet('<front>');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/environment_indicator.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $session->pageTextContains('Development Environment');
  }

  /**
   * Tests the indicator with the default configuration.
   */
  public function testIndicatorDefaultConfiguration(): void {
    $this->drupalLogin($this->privilegedUser);
    $this->state->set('environment_indicator.current_release', 'v1.2.44');
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Indicator Environment')
      ->set('fg_color', '#000000')
      ->set('bg_color', '#ffffff')
      ->save();
    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);
    $this->drupalGet('<front>');
    $this->assertSession()->pageTextNotContains('v1.2.44');
    $this->assertSession()->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $this->assertSession()->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/environment_indicator.js']");
    $this->assertSession()->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $this->assertSession()->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $this->assertSession()->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");
    $this->assertSession()->pageTextContains('Indicator Environment');
    $page = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('environmentIndicator', $page);
    $this->assertStringContainsString('<style data-environment-indicator-styles>html body {
      --environment-indicator-color: #000000;
      --environment-indicator-background-color: #ffffff;
    }</style>', $page);
    $this->drupalLogin($this->currentReleaseUser);
    $this->assertSession()->pageTextContains('v1.2.44');
    $this->assertSession()->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $this->assertSession()->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/environment_indicator.js']");
    $this->assertSession()->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $this->assertSession()->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $this->assertSession()->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");

  }

  /**
   * Tests that a user without access permission sees no indicator.
   */
  public function testUserWithoutEnvironmentIndicatorPermission(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Restricted Environment')
      ->set('fg_color', '#abcdef')
      ->set('bg_color', '#123456')
      ->save();

    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);

    // User only has toolbar access, no environment indicator access.
    $user = $this->drupalCreateUser(['access toolbar']);
    $this->drupalLogin($user);
    $this->drupalGet('<front>');

    $session = $this->assertSession();

    // They should not see the environment name.
    $session->pageTextNotContains('Restricted Environment');

    // They should not see any indicator-related libraries or selectors.
    $session->elementNotExists('css', '#environment-indicator');
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorModulePath}/css/environment_indicator.css']");
    $session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/environment_indicator.js']");
    $session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementNotExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $session->elementNotExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");

    $page = $this->getSession()->getPage()->getContent();
    $this->assertStringNotContainsString('environmentIndicator', $page);
    $this->assertStringNotContainsString('<style data-environment-indicator-styles>html body {
      --environment-indicator-color: #abcdef;
      --environment-indicator-background-color: #123456;
    }</style>', $page);

  }

  /**
   * Tests that a user with access permission sees the indicator.
   */
  public function testUserWithEnvironmentIndicatorPermission(): void {
    $config = $this->config('environment_indicator.indicator');
    $config->set('name', 'Restricted Environment')
      ->set('fg_color', '#abcdef')
      ->set('bg_color', '#123456')
      ->save();

    $this->container->get('cache_tags.invalidator')->invalidateTags(['config:environment_indicator.indicator']);

    // User has both toolbar and environment indicator access.
    $user = $this->drupalCreateUser(['access toolbar', 'access environment indicator']);
    $this->drupalLogin($user);
    $this->drupalGet('<front>');

    $session = $this->assertSession();

    // They should see the environment name.
    $session->pageTextContains('Restricted Environment');

    // They should see all indicator-related libraries or selectors.
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/tinycon.min.js']");
    $session->elementExists('css', "script[src*='{$this->environmentIndicatorModulePath}/js/favicon.js']");
    $session->elementExists('css', "link[href*='{$this->environmentIndicatorToolbarModulePath}/css/toolbar.css']");

    $page = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('environmentIndicator', $page);
    $this->assertStringContainsString('<style data-environment-indicator-styles>html body {
      --environment-indicator-color: #abcdef;
      --environment-indicator-background-color: #123456;
    }</style>', $page);
  }

}
