/**
 * EPT sticky-menu scroll spy.
 *
 * Highlights the menu link whose section is centred in the viewport.
 *
 * Requires:
 *   - core/drupal
 *   - core/once
 */
(function (Drupal, once) {
  'use strict';

  const MENU_PARAGRAPH = '.paragraph--type--ept-sticky-menu';
  const OFFSET_EXTRA   = 10;                 // extra breathing room
  const MOBILE_BREAK   = 960;

  /**
   * Highlight active section in menu.
   */
  Drupal.behaviors.eptStickyMenuScrollSpy = {
    attach(context) {
      // Run exactly once per full page load.
      const menu = once('ept-scroll-spy', MENU_PARAGRAPH, context).shift();
      if (!menu) {
        return;
      }

      const links = menu.querySelectorAll('a[href^="#"]');
      if (!links.length) {
        return;
      }

      const sections = Array.from(links)
        .map(link => document.getElementById(link.getAttribute('href').substring(1)))
        .filter(Boolean);

      const observer = new IntersectionObserver((entries) => {
        entries.forEach((entry) => {
          if (!entry.isIntersecting) {
            return;
          }

          // 1️⃣ Section observed.
          const id         = entry.target.id;
          const activeLink = menu.querySelector(`a[href="#${id}"]`);

          if (activeLink) {
            links.forEach(l => l.classList.remove('is-active'));
            activeLink.classList.add('is-active');

            /* NEW ➜ put active section name into the mobile toggle */
            const label = menu.querySelector('.ept-mobile-toggle__label');
            if (label) { label.textContent = activeLink.textContent.trim(); }

            // 2️⃣ Section now active.
          }
        });
      }, {
        rootMargin: '-50% 0px -50% 0px',
        threshold: 0
      });

      sections.forEach(section => observer.observe(section));
    }
  };

  /**
   * Click on sticky menu link.
   */
  Drupal.behaviors.eptStickyMenuScrollOffset = {
    attach(context) {
      const menu = once('ept-offset-menu', MENU_PARAGRAPH, context).shift();
      if (!menu) {
        return;
      }

      const links = menu.querySelectorAll('a[href^="#"]');
      if (!links.length) {
        return;
      }

      // Attach the click handler just once per link.
      once('ept-offset-link', links).forEach(link => {
        link.addEventListener('click', (e) => {
          const hash = link.getAttribute('href');
          const id   = hash.slice(1);
          const target = document.getElementById(id);

          if (!target) {
            return; // broken anchor
          }

          e.preventDefault();              // stop the browser’s default jump

          const menuH = menu.offsetHeight;
          const y     = target.getBoundingClientRect().top +
            window.scrollY -
            menuH -
            OFFSET_EXTRA;

          window.scrollTo({ top: y, behavior: 'smooth' });

          // Keep the URL hash in sync without extra jump.
          history.pushState(null, '', hash);
        });
      });

      /* ---------- handle page-load with hash ---------- */
      if (context === document && window.location.hash) {
        const id     = window.location.hash.slice(1);
        const target = document.getElementById(id);
        const menuH  = menu.offsetHeight;

        if (target) {
          // Delay until layout is stable, then nudge up.
          window.requestAnimationFrame(() => {
            const y = target.getBoundingClientRect().top +
              window.scrollY -
              menuH -
              OFFSET_EXTRA;
            window.scrollTo({ top: y });
          });
        }
      }
    }
  };

  /**
   * Sticky-js menu blocks.
   */
  Drupal.behaviors.eptStickyMenuBlocks = {
    attach(context) {
      if (once('sticky-js-init', 'body', context).length) {
        // Pass a selector string — NOT a DOM node.
        // Sticky-js watches resize/scroll so one instance is enough.
        // eslint-disable-next-line no-new
        new Sticky('.sticky-js-block');
      }
    }
  };

  Drupal.behaviors.eptStickyMenuMobileToggle = {
    attach(context) {
      // Initialise once per menu.
      const menu = once('ept-mobile-toggle', MENU_PARAGRAPH, context).shift();
      if (!menu) {
        return;
      }

      /* ---------- build the toggle span ---------- */
      const toggle = document.createElement('span');
      toggle.className = 'ept-mobile-toggle';
      // basic a11y so a <span> behaves like a button
      toggle.setAttribute('role', 'button');
      toggle.setAttribute('tabindex', '0');
      toggle.innerHTML = `
        <span class="ept-mobile-toggle__label">Menu</span>
        <span class="ept-mobile-toggle__icon"></span>
      `;
      // Insert before the links container.
      const linksContainer = menu.querySelector('.field--name-field-ept-sticky-menu-links');
      linksContainer.parentNode.insertBefore(toggle, linksContainer);

      /* ---------- interaction ---------- */
      const openMenu   = () => {
        menu.classList.add('is-expanded');
        toggle.classList.add('ept-mobile-toggle--open');
        toggle.setAttribute('aria-expanded', 'true');
      };

      const closeMenu  = () => {
        menu.classList.remove('is-expanded');
        toggle.classList.remove('ept-mobile-toggle--open');
        toggle.setAttribute('aria-expanded', 'false');
      };

      const toggleMenu = () => {
        menu.classList.contains('is-expanded') ? closeMenu() : openMenu();
      };

      toggle.addEventListener('click', toggleMenu);

      toggle.addEventListener('keydown', (e) => {
        if (e.key === ' ' || e.key === 'Enter') {
          e.preventDefault();
          menu.classList.contains('is-expanded') ? closeMenu() : openMenu();
        }
      });

      /* ---------- auto-close on link click (mobile only) ---------- */
      linksContainer.querySelectorAll('a[href^="#"]').forEach(link => {
        link.addEventListener('click', () => {
          if (window.innerWidth <= MOBILE_BREAK) {
            closeMenu();
          }
        });
      });

      /* ---------- handle resize ---------- */
      const mq = window.matchMedia(`(min-width: ${MOBILE_BREAK + 1}px)`);
      mq.addEventListener('change', (e) => {
        if (e.matches) {
          // Switched to desktop → ensure menu visible & toggle reset.
          menu.classList.remove('is-expanded');
          toggle.classList.remove('ept-mobile-toggle--open');
        }
      });
    }
  };

})(Drupal, once);