<?php

declare(strict_types=1);

namespace Drupal\es_attachment\EventSubscriber;

use Drupal\elasticsearch_connector\Event\IndexParamsEvent;
use Drupal\es_attachment\Helpers;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Actions during the indexing process of one element.
 */
class IndexItemEvent implements EventSubscriberInterface {

  /**
   * The Helpers service.
   *
   * @var \Drupal\es_attachment\Helpers
   */
  private Helpers $helpers;

  /**
   * Constructs a new QueryEvent object.
   *
   * @param \Drupal\es_attachment\Helpers $helpers
   *   The Helpers service.
   */
  public function __construct(Helpers $helpers) {
    $this->helpers = $helpers;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[IndexParamsEvent::class][] = ['pipelineProcessing', 101];
    return $events;
  }

  /**
   * Processes the index parameters before indexing an item.
   *
   * This function modifies the index parameters to include the necessary
   * transformations for file content and adds the specified pipeline.
   *
   * @param \Drupal\elasticsearch_connector\Event\IndexParamsEvent $event
   *   The event object containing data to index parameters.
   */
  public function pipelineProcessing(IndexParamsEvent $event): void {

    $params = $event->getParams();

    // #1 : Add "file_content" key to each file content.
    $index = $this->helpers->loadIndex($event->getOriginalIndexId());

    $attachment_processor_enable = $this->helpers->processorEnable($index);
    if (!$attachment_processor_enable) {
      return;
    }

    $fields = $this->helpers->getDocumentFields($index);
    foreach ($params['body'] as $key => $content) {
      foreach ($content as $field_name => $item) {
        if (in_array($field_name, $fields)) {
          $new_content = [];
          foreach ($item as $fileContent) {
            $new_content[] = ['data' => $fileContent];
          }
          $params['body'][$key][$field_name] = $new_content;
          $params['body'][$key - 1]['index']['pipeline'] = 'es_attachment';
        }
      }

    }

    $event->setParams($params);
  }

}
