<?php

declare(strict_types=1);

namespace Drupal\es_attachment\EventSubscriber;

use Drupal\elasticsearch_connector\Event\QueryParamsEvent;
use Drupal\es_attachment\Helpers;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Actions during the indexing process of one element.
 */
class QueryEvent implements EventSubscriberInterface {

  /**
   * The Helpers service.
   *
   * @var \Drupal\es_attachment\Helpers
   */
  private Helpers $helpers;

  /**
   * Constructs a new QueryEvent object.
   *
   * @param \Drupal\es_attachment\Helpers $helpers
   *   The Helpers service.
   */
  public function __construct(Helpers $helpers) {
    $this->helpers = $helpers;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[QueryParamsEvent::class][] = ['alterQuery', 90];
    return $events;
  }

  /**
   * Add Attachment to search queries.
   *
   * @param \Drupal\elasticsearch_connector\Event\QueryParamsEvent $event
   *   The QueryParamsEvent object.
   */
  public function alterQuery(QueryParamsEvent $event): void {

    // Check if there is the "no_attachments" flag.
    $query = $event->getQuery();
    $no_attachments = $query->getOption("no_attachments");
    if (isset($no_attachments) && $no_attachments === TRUE) {
      return;
    }

    // Get the index name.
    $index_name = $query->getIndex()->getOriginalId();

    // Check if the index is using the attachment processor.
    $index = $this->helpers->loadIndex($index_name);
    $attachment_processor_enable = $this->helpers->processorEnable($index);
    $params = $event->getParams();

    // Only alter if processor is enabled.
    if (!$attachment_processor_enable) {
      return;
    }

    // Adapt Match method to parseMode.
    $parse_mode = $query->getParseMode()->getPluginId();

    // Parse mode "direct as default.
    $query_keys = $query->getKeys();
    $queryString = $query_keys;

    if (is_array($query_keys)) {
      $queryString = $query_keys[0];
    }
    if ($parse_mode == 'terms') {
      $keys = $query_keys;
      if (isset($keys['#conjunction'])) {
        unset($keys['#conjunction']);
      }
      $queryString = implode(' ', $keys);
    }

    // Get Conjuction.
    $conjunction = 'and';
    if ($parse_mode != 'direct') {
      $conjunction = $query_keys['#conjunction'];
    }

    // Only alter if it's a search query.
    if (trim($queryString) == '') {
      return;
    }

    $fields = $this->helpers->getDocumentFields($index);
    // If no fields are found, return.
    if (count($fields) === 0) {
      return;
    }

    $originalBoolQuery = $params['body']['query'];
    if (isset($params['body']['query']['bool']['must'])) {
      $originalBoolQuery = $params['body']['query']['bool']['must'];
    }

    unset($params['body']['query']);

    // Find a way to add boost.
    $boost = 1;

    // We need to change the bool query from must to should.
    // This is required to add support for nested and string queries.
    // Add min match param.
    $params['body']['query']['bool']['minimum_should_match'] = 1;
    $params['body']['query']['bool']['should'] = [];
    $params['body']['query']['bool']['should'][] = $originalBoolQuery;

    // Preparer excluded fields.
    if (!isset($params['body']['_source']['excludes'])) {
      $params['body']['_source']['excludes'] = [];
    }

    // Build nestedQuery.
    // @see https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-nested-query.html.
    foreach ($fields as $field_name) {
      // Exclude the file content from query response.
      $params['body']['_source']['excludes'][] = $field_name;

      $match = [
        'match' => [
          $field_name . '.attachment.content' => [
            'query' => $queryString,
            'operator' => $conjunction,
          ],
        ],
      ];

      if ($parse_mode == 'phrase') {
        $match = [
          'match_phrase' => [
            $field_name . '.attachment.content' => [
              'query' => $queryString,
            ],
          ],
        ];
      }

      $params['body']['query']['bool']['should'][] = [
        'nested' => [
          'path' => $field_name,
          'query' => [
            'constant_score' => [
              'filter' => $match,
              'boost' => $boost,
            ],
          ],
        ],
      ];
    }

    // Replace query.
    $event->setParams($params);
  }

}
