<?php

declare(strict_types=1);

namespace Drupal\es_attachment;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\IndexInterface;
use Drupal\search_api_attachments\Plugin\search_api\processor\FilesExtractor;

/**
 * Helpers for attachment processing.
 */
final class Helpers {

  /**
   * Entity type manager.
   *
   * @var Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Search_API configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Constructs a Helpers object.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    // Add your dependencies here:
    $this->entityTypeManager = $entityTypeManager;
    $this->config = $configFactory->get('search_api_attachments.admin_config');
  }

  /**
   * Provides the Search API Index name.
   */
  public function getDocumentFields(IndexInterface $index):array {

    /** @var \Drupal\search_api\Item\Field[] $fields */
    $fields = $index->getFields();

    $extractor_fields = [];
    foreach ($fields as $field_name => $field) {
      if (substr($field_name, 0, 4) == FilesExtractor::SAA_PREFIX) {
        $extractor_fields[] = $field->getFieldIdentifier();
      }
    }
    return $extractor_fields;
  }

  /**
   * Check if the es_pipeline_extractor processor is enabled.
   *
   * @param \Drupal\search_api\Entity\IndexInterface $index
   *   The Search API Index.
   *
   * @return bool
   *   True if processor is enabled, false otherwise.
   */
  public function processorEnable(IndexInterface $index):bool {

    $processors = $index->getProcessors();

    $processorsEnable = FALSE;
    if (
      isset($processors['file_attachments']) &&
      $processors['file_attachments'] instanceof FilesExtractor) {

      if ($this->config->get('extraction_method') == 'es_pipeline_extractor') {
        $processorsEnable = TRUE;
      }
    }
    return $processorsEnable;
  }

  /**
   * Load Search API Index.
   *
   * @param string $indexName
   *   The name of the index.
   *
   * @return \Drupal\search_api\Entity\Index|null
   *   The index.
   */
  public function loadIndex(string $indexName):?Index {
    try {
      /** @var \Drupal\search_api\Entity\Index $index */
      $index = $this->entityTypeManager->getStorage('search_api_index')->load($indexName);
    }
    catch (EntityStorageException $e) {

    }

    return $index;
  }

}
