<?php

declare(strict_types=1);

namespace Drupal\es_attachment\EventSubscriber;

use Elastic\Elasticsearch\Client;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Drupal\elasticsearch_connector\Event\AlterSettingsEvent;
use Drupal\es_attachment\Helpers;
use Drupal\search_api\IndexInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Manage pipeline settings.
 */
class UpdateIndexSettingsEvent implements EventSubscriberInterface {

  /**
   * Pipeline name to be used for file attachments.
   *
   * @var string
   */
  protected $pipelineName = 'es_attachment';

  /**
   * The Helpers service.
   *
   * @var \Drupal\es_attachment\Helpers
   */
  private Helpers $helpers;

  /**
   * Constructs a new QueryEvent object.
   *
   * @param \Drupal\es_attachment\Helpers $helpers
   *   The Helpers service.
   */
  public function __construct(Helpers $helpers) {
    $this->helpers = $helpers;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[AlterSettingsEvent::class][] = ['managePipeline', 999];
    return $events;
  }

  /**
   * Manage the Elasticsearch pipeline.
   *
   * @param \Drupal\elasticsearch_connector\Event\AlterSettingsEvent $event
   *   The AlterSettingsEvent object containing the index settings.
   */
  public function managePipeline(AlterSettingsEvent $event): void {
    /** @var \Drupal\search_api\IndexInterface $index */
    $index = $event->getIndex();

    $need_pipeline = $this->helpers->processorEnable($index);

    if ($need_pipeline) {
      $this->setupPipeline($index);
    }
    else {
      $this->unSetupPipeline($index);
    }
  }

  /**
   * Sets up the Elasticsearch pipeline for the given index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index for which the pipeline needs to be set up.
   */
  private function setupPipeline(IndexInterface $index) {
    $backend = $index->getServerInstance()->getBackend();
    /** @var \Elastic\Elasticsearch\Client $client */
    $client = $backend->getClient();

    try {
      // Try to get pipeline to check if it already exists.
      $client->ingest()->getPipeline(['id' => $this->pipelineName]);
      // Delete pipeline.
      $client->ingest()->deletePipeline(['id' => $this->pipelineName]);
    }
    catch (ClientResponseException $e) {
      // The pipeline does not exist.
    }

    // Add pipeline.
    $fields = $this->helpers->getDocumentFields($index);
    $this->addPipeline($client, $fields);
  }

  /**
   * Unsets up the Elasticsearch pipeline for the given index.
   *
   * @param \Drupal\search_api\IndexInterface $index
   *   The index for which the pipeline needs to be unset up.
   */
  private function unSetupPipeline(IndexInterface $index) {
    $backend = $index->getServerInstance()->getBackend();
    /** @var \Elastic\Elasticsearch\Client $client */
    $client = $backend->getClient();

    try {
      // Try to get pipeline to check if it already exists.
      $client->ingest()->getPipeline(['id' => $this->pipelineName]);
    }
    catch (ClientResponseException $e) {
      return;
    }

    // Delete pipeline.
    $client->ingest()->deletePipeline(['id' => $this->pipelineName]);
  }

  /**
   * Adds a pipeline to Elasticsearch for extracting attachment.
   *
   * @param \Elastic\Elasticsearch\Client $client
   *   The Elasticsearch client to use for making API requests.
   * @param array $fields
   *   An array of fields from the index where attachments are expected.
   */
  private function addPipeline(Client $client, array $fields) {

    $processors = [];
    foreach ($fields as $field) {

      $processors[] = [
        "foreach" => [
          "field" => $field,
          "processor" => [
            "attachment" => [
              "target_field" => "_ingest._value.attachment",
              "field" => "_ingest._value.data",
              "indexed_chars" => -1,
              "remove_binary" => TRUE,
              "properties" => ['content'],
              "ignore_missing" => TRUE,
            ],
          ],
        ],
      ];

    }

    // dsm($processors);
    $params = [];
    $params['id'] = $this->pipelineName;
    $params['body'] = [
      'description' => 'Extract attachment information from arrays',
      'processors' => $processors,
    ];
    $client->ingest()->putPipeline($params);

  }

}
