<?php

namespace Drupal\es_filter_analyser\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\es_filter_analyser\Entity\AnalyserInterface;

/**
 * Edition Form for Analyser Entity.
 */
class AnalyserForm extends EntityForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs an AnalyserForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\es_filter_analyser\Entity\AnalyserInterface $analyser */
    $analyser = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $analyser->label(),
      '#description' => $this->t("Label for the Analyser."),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $analyser->id(),
      '#machine_name' => [
        'exists' => '\Drupal\es_filter_analyser\Entity\Analyser::load',
      ],
      '#disabled' => !$analyser->isNew(),
    ];

    $form['tokenizer'] = [
      '#type' => 'select',
      '#title' => $this->t('Tokenizer'),
      '#default_value' => $analyser->getTokenizer() ?? 'Standard',
      '#description' => $this->t('Select the tokenizer to use for this analyser.'),
      '#required' => TRUE,
      '#options' => $this->getTokenizerOptions(),
    ];

    // Filters with drag-and-drop ordering.
    $form['filters'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Filter'),
        $this->t('Weight'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No filters added yet.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'filter-weight',
        ],
      ],
      '#prefix' => '<div id="filters-wrapper">',
      '#suffix' => '</div>',
    ];

    $filters = $form_state->getValue('filters');
    if ($filters === NULL) {
      $filters = $analyser->getFilters();
    }

    foreach ($filters as $weight => $filter_id) {
      $filter_entity = $this->entityTypeManager->getStorage('es_filter')->load($filter_id);
      if ($filter_entity) {
        $form['filters'][$filter_id] = $this->buildFilterRow($filter_id, $filter_entity, $weight);
      }
    }

    // Add new filter field.
    $form['add_filter'] = [
      '#type' => 'details',
      '#title' => $this->t('Add filter'),
      '#open' => TRUE,
    ];

    $form['add_filter']['new_filter'] = [
      '#type' => 'select',
      '#title' => $this->t('Filter'),
      '#description' => $this->t('Select a filter to add.'),
      '#options' => $this->getAvailableFiltersOptions($analyser),
      '#empty_option' => $this->t('- Select a filter -'),
    ];

    $form['add_filter']['add_filter_button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add filter'),
      '#submit' => ['::addFilterSubmit'],
      '#ajax' => [
        'callback' => '::updateFilterAjax',
        'wrapper' => 'filters-wrapper',
      ],
    ];

    return $form;
  }

  /**
   * Get available filters options for the select list.
   *
   * @param \Drupal\es_filter_analyser\Entity\AnalyserInterface $analyser
   *   The analyser entity.
   *
   * @return array
   *   An array of filter options keyed by filter ID.
   */
  protected function getAvailableFiltersOptions(AnalyserInterface $analyser) {
    $options = [];
    $current_filters = $analyser->getFilters();

    // Load all filter entities.
    $filters = $this->entityTypeManager
      ->getStorage('es_filter')
      ->loadMultiple();

    foreach ($filters as $filter_id => $filter) {
      // Only show filters that are not already added.
      if (!isset($current_filters[$filter_id])) {
        $options[$filter_id] = $filter->label() . ' (' . $filter->getType() . ')';
      }
    }

    return $options;
  }

  /**
   * Build a filter row.
   *
   * @param string $filter_id
   *   The filter ID.
   * @param \Drupal\es_filter_analyser\Entity\FilterInterface $filter_entity
   *   The filter entity.
   * @param int $weight
   *   The weight.
   *
   * @return array
   *   The filter row render array.
   */
  protected function buildFilterRow($filter_id, $filter_entity, $weight) {
    return [
      '#attributes' => [
        'class' => ['draggable'],
      ],
      'filter' => [
        '#plain_text' => $filter_entity->label() . ' (' . $filter_entity->getType() . ')',
      ],
      'weight' => [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @title', ['@title' => $filter_entity->label()]),
        '#title_display' => 'invisible',
        '#default_value' => $weight,
        '#attributes' => ['class' => ['filter-weight']],
      ],
      'operations' => [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_filter_' . $filter_id,
        '#submit' => ['::removeFilterSubmit'],
        '#filter_id' => $filter_id,
        '#ajax' => [
          'callback' => '::updateFilterAjax',
          'wrapper' => 'filters-wrapper',
        ],
      ],
    ];
  }

  /**
   * Submit handler for adding a filter.
   */
  public function addFilterSubmit(array &$form, FormStateInterface $form_state) {
    $new_filter_id = $form_state->getValue('new_filter');
    if ($new_filter_id) {
      /** @var \Drupal\es_filter_analyser\Entity\AnalyserInterface $analyser */
      $analyser = $this->entity;
      $filters = $analyser->getFilters();

      if (!isset($filters[$new_filter_id])) {
        $max_weight = 0;
        foreach ($filters as $filter_data) {
          if (($filter_data['weight'] ?? 0) > $max_weight) {
            $max_weight = $filter_data['weight'];
          }
        }

        $filters[$new_filter_id] = [
          'weight' => $max_weight + 1,
        ];
        $form_state->setValue('filters', $filters);
      }

      $form_state->setValue('new_filter', '');
    }

    $form_state->setRebuild();
  }

  /**
   * Ajax callback for adding a filter.
   *
   * Returns the updated filters table element to be replaced via AJAX
   * after a new filter has been added to the analyser.
   *
   * @param array $form
   *   The complete form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The filters table render array to replace the existing table.
   */
  public function updateFilterAjax(array &$form, FormStateInterface $form_state) {
    return $form['filters'];
  }

  /**
   * Submit handler for removing a filter.
   */
  public function removeFilterSubmit(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $filter_id = $triggering_element['#filter_id'];
    $filters = $form_state->getValue('filters');

    if (isset($filters[$filter_id])) {
      unset($filters[$filter_id]);
      $form_state->setValue('filters', $filters);
    }
    $form_state->setRebuild();
  }

  /**
   * Get available Elasticsearch tokenizers.
   *
   * @return array
   *   An array of tokenizer options.
   */
  protected function getTokenizerOptions() {
    return [
      // Word Oriented Tokenizers.
      'standard' => $this->t('Standard - Grammar based tokenizer'),
      'letter' => $this->t('Letter - Breaks on non-letter characters'),
      'lowercase' => $this->t('Lowercase - Like letter but lowercases'),
      'whitespace' => $this->t('Whitespace - Breaks on whitespace'),
      'uax_url_email' => $this->t('UAX URL Email - Like standard but preserves URLs and emails'),

      // Partial Word Tokenizers.
      'ngram' => $this->t('N-Gram - Breaks text into words on specified characters'),
      'edge_ngram' => $this->t('Edge N-Gram - Breaks text into words on specified characters from the beginning'),

      // Structured Text Tokenizers.
      'keyword' => $this->t('Keyword - No-op tokenizer (entire input as single token)'),
      'pattern' => $this->t('Pattern - Uses regex to split text'),
      'simple_pattern' => $this->t('Simple Pattern - Uses Lucene regex to split text'),
      'char_group' => $this->t('Char Group - Breaks on defined character sets'),
      'simple_pattern_split' => $this->t('Simple Pattern Split - Uses Lucene regex to split on separator'),
      'path_hierarchy' => $this->t('Path Hierarchy - Breaks hierarchical values (file paths)'),

      // Language-Specific Tokenizers.
      'thai' => $this->t('Thai - Thai language tokenizer'),
      'icu_tokenizer' => $this->t('ICU Tokenizer - Unicode Text Segmentation'),
      'kuromoji_tokenizer' => $this->t('Kuromoji - Japanese language tokenizer'),
      'nori_tokenizer' => $this->t('Nori - Korean language tokenizer'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\es_filter_analyser\Entity\AnalyserInterface $analyser */
    $analyser = $this->entity;

    // Process filters with weights from the table.
    $filters_table = $form_state->getValue('filters', []);

    if (!is_array($filters_table)) {
      $filters_table = [];
    }
    $filters = [];
    foreach ($filters_table as $filter_id => $filter_data) {
      if (is_array($filter_data) && isset($filter_data['weight'])) {
        $filters[ $filter_data['weight'] ] = $filter_id;
      }
    }
    ksort($filters);
    $analyser->setFilters(array_values($filters));

    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        \SAVED_NEW => $this->t('Created new analyser %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated analyser %label.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

}
