<?php

namespace Drupal\es_filter_analyser\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\es_filter_analyser\Plugin\FilterTypeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Edition Form for Filter Entity.
 */
class FilterForm extends EntityForm {

  /**
   * The filter type plugin manager.
   *
   * @var \Drupal\es_filter_analyser\Plugin\FilterTypeManager
   */
  protected $filterTypeManager;

  /**
   * Constructs a FilterForm object.
   *
   * @param \Drupal\es_filter_analyser\Plugin\FilterTypeManager $filter_type_manager
   *   The filter type plugin manager.
   */
  public function __construct(FilterTypeManager $filter_type_manager) {
    $this->filterTypeManager = $filter_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.filter_type')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\es_filter_analyser\Entity\FilterInterface $filter */
    $filter = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $filter->label(),
      '#description' => $this->t("Label for the Filter."),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $filter->id(),
      '#machine_name' => [
        'exists' => '\Drupal\es_filter_analyser\Entity\Filter::load',
      ],
      '#disabled' => !$filter->isNew(),
    ];

    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Filter Type'),
      '#default_value' => $filter->getType(),
      '#description' => $this->t('Select the type of filter.'),
      '#required' => TRUE,
      '#options' => $this->filterTypeManager->getFilterTypeOptions(),
      '#ajax' => [
        'callback' => '::updateSettingsForm',
        'wrapper' => 'filter-settings-wrapper',
      ],
    ];

    // Container for plugin-specific settings.
    $form['settings_container'] = [
      '#type' => 'container',
      '#prefix' => '<div id="filter-settings-wrapper">',
      '#suffix' => '</div>',
    ];

    // Get the selected filter type.
    $selected_type = $form_state->getValue('type') ?? $filter->getType();

    if ($selected_type) {
      try {
        /** @var \Drupal\es_filter_analyser\Plugin\FilterTypeInterface $plugin */
        $plugin = $this->filterTypeManager->createInstance($selected_type);
        $current_settings = $filter->getSettings();

        $form['settings_container']['settings'] = $plugin->buildConfigurationForm(
          [],
          $form_state,
          $current_settings
        );

        $form['settings_container']['settings']['#tree'] = TRUE;
      }
      catch (\Exception $e) {
        $form['settings_container']['error'] = [
          '#markup' => $this->t('Error loading filter type configuration: @message', ['@message' => $e->getMessage()]),
        ];
      }
    }

    // Hidden field to store settings as JSON (for compatibility).
    $form['settings_json'] = [
      '#type' => 'hidden',
      '#default_value' => !empty($filter->getSettings()) ? json_encode($filter->getSettings()) : '',
    ];

    return $form;
  }

  /**
   * Ajax callback to update the settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The settings container element.
   */
  public function updateSettingsForm(array &$form, FormStateInterface $form_state) {
    return $form['settings_container'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $selected_type = $form_state->getValue('type');
    if ($selected_type) {
      try {
        if (isset($form['settings_container']['settings'])) {
          /** @var \Drupal\es_filter_analyser\Plugin\FilterTypeInterface $plugin */
          $plugin = $this->filterTypeManager->createInstance($selected_type);
          $plugin->validateConfigurationForm($form['settings_container']['settings'], $form_state);
        }
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('type', $this->t('Error validating filter type: @message', ['@message' => $e->getMessage()]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\es_filter_analyser\Entity\FilterInterface $filter */
    $filter = $this->entity;

    $selected_type = $form_state->getValue('type');
    if ($selected_type) {
      try {
        /** @var \Drupal\es_filter_analyser\Plugin\FilterTypeInterface $plugin */
        $plugin = $this->filterTypeManager->createInstance($selected_type);
        $settings = $plugin->processPluginSettings($form['settings_container']['settings'], $form_state);
        $filter->setSettings($settings);
      }
      catch (\Exception $e) {
        $this->messenger()->addError($this->t('Error saving filter settings: @message', ['@message' => $e->getMessage()]));
        return;
      }
    }
    else {
      $filter->setSettings([]);
    }

    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        \SAVED_NEW => $this->t('Created new filter %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated filter %label.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

}
