<?php

namespace Drupal\es_filter_analyser\Plugin\FilterType;

use Drupal\es_filter_analyser\Plugin\FilterTypeBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a Common Grams filter type.
 *
 * @FilterType(
 *   id = "common_grams",
 *   label = @Translation("Common Grams"),
 *   description = @Translation("Generates bigrams for frequently occurring terms.")
 * )
 */
class CommonGrams extends FilterTypeBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'common_words' =>
        [
          'a', 'an', 'and', 'are', 'as', 'at', 'be', 'but', 'by', 'for', 'if', 'in',
          'into', 'is', 'it', 'no', 'not', 'of', 'on', 'or', 'such', 'that', 'the',
          'their', 'then', 'there', 'these', 'they', 'this', 'to', 'was', 'will', 'with',
        ],
      'ignore_case' => FALSE,
      'query_mode' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, array $configuration = []) {
    $configuration += $this->defaultConfiguration();

    $form['common_words'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Common words'),
      '#default_value' => implode("\n", $configuration['common_words']),
      '#description' => $this->t('Enter one common word per line.'),
      '#required' => TRUE,
      '#rows' => 10,
    ];

    $form['ignore_case'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Ignore case'),
      '#default_value' => $configuration['ignore_case'],
      '#description' => $this->t('If checked, matching is case-insensitive.'),
    ];

    $form['query_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Query mode'),
      '#default_value' => $configuration['query_mode'],
      '#description' => $this->t('If checked, excludes common tokens from output.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array $values) {
    $common_words = $values['common_words'];
    $common_words_array = array_filter(array_map('trim', explode("\n", $common_words)));

    return [
      'common_words' => array_values($common_words_array),
      'ignore_case' => (bool) $values['ignore_case'],
      'query_mode' => (bool) $values['query_mode'],
    ];
  }

}
