<?php

namespace Drupal\es_filter_analyser\Plugin\FilterType;

use Drupal\es_filter_analyser\Plugin\FilterTypeBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a Length filter type.
 *
 * @FilterType(
 *   id = "length",
 *   label = @Translation("Length"),
 *   description = @Translation("Removes tokens that are too short or too long.")
 * )
 */
class Length extends FilterTypeBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'min' => 0,
      'max' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, array $configuration = []) {
    $configuration += $this->defaultConfiguration();

    $form['min'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum length'),
      '#default_value' => $configuration['min'],
      '#description' => $this->t('Tokens shorter than this will be removed. Set to 0 for no minimum.'),
      '#min' => 0,
    ];

    $form['max'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum length'),
      '#default_value' => $configuration['max'],
      '#description' => $this->t('Tokens longer than this will be removed. Leave empty for no maximum.'),
      '#min' => 1,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $min = $form_state->getValue('min');
    $max = $form_state->getValue('max');

    if ($max !== '' && $max !== NULL && $min > $max) {
      $form_state->setErrorByName('max', $this->t('Maximum length must be greater than or equal to minimum length.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array $values) {
    $config = [
      'min' => (int) $values['min'],
    ];

    $max = $values['max'];
    if ($max !== '' && $max !== NULL) {
      $config['max'] = (int) $max;
    }

    return $config;
  }

}
