<?php

namespace Drupal\es_filter_analyser\Plugin\search_api\processor;

use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\search_api\Plugin\PluginFormTrait;
use Drupal\search_api\Processor\ProcessorPluginBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\search_api\DataType\DataTypePluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configures custom analysers for Search API fields.
 *
 * @SearchApiProcessor(
 *   id = "analyser_processor",
 *   label = @Translation("ElasticSearch Analyser"),
 *   description = @Translation("Allows configuring custom Elasticsearch analysers for index and search operations on each field."),
 *   stages = {
 *     "pre_index_save" = 0,
 *     "preprocess_query" = 0,
 *   }
 * )
 */
class AnalyserProcessor extends ProcessorPluginBase implements PluginFormInterface {

  use PluginFormTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The data type plugin manager.
   *
   * @var \Drupal\search_api\DataType\DataTypePluginManager
   */
  protected $dataTypePluginManager;

  /**
   * Constructs an AnalyserProcessor object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\search_api\DataType\DataTypePluginManager $data_type_plugin_manager
   *   The data type plugin manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, DataTypePluginManager $data_type_plugin_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->dataTypePluginManager = $data_type_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.search_api.data_type')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'global_search_analyser' => '',
      'field_analysers' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $index = $this->index;
    $fields = $index->getFields();
    $configuration = $this->getConfiguration();

    $analyser_options = $this->getAnalyserOptions();

    // Global search analyser configuration.
    $form['global_search_analyser'] = [
      '#type' => 'select',
      '#title' => $this->t('Global Search Analyser'),
      '#description' => $this->t('Select a global analyser to be used for all search queries. Field-specific analysers will override this setting.'),
      '#options' => $analyser_options,
      '#default_value' => $configuration['global_search_analyser'] ?? '',
      '#empty_option' => $this->t('- None -'),
    ];

    $form['field_analysers'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Field'),
        $this->t('Type'),
        $this->t('Index Analyser'),
        $this->t('Search Analyser'),
      ],
      '#empty' => $this->t('No fields available.'),
    ];

    foreach ($fields as $field_id => $field) {
      $field_type = $field->getType();
      $field_label = $field->getLabel();

      // Only show text fields.
      if (!$this->isTextField($field_type)) {
        continue;
      }

      $current_config = $configuration['field_analysers'][$field_id] ?? [];

      $form['field_analysers'][$field_id]['field_info'] = [
        '#markup' => '<strong>' . $field_label . '</strong><br><small>' . $field_id . '</small>',
      ];

      $form['field_analysers'][$field_id]['type'] = [
        '#markup' => $this->getFieldTypeLabel($field_type),
      ];

      $form['field_analysers'][$field_id]['index_analyser'] = [
        '#type' => 'select',
        '#title' => $this->t('Index Analyser for @field', ['@field' => $field_label]),
        '#title_display' => 'invisible',
        '#options' => $analyser_options,
        '#default_value' => $current_config['index_analyser'] ?? '',
        '#empty_option' => $this->t('- Default -'),
      ];

      $form['field_analysers'][$field_id]['search_analyser'] = [
        '#type' => 'select',
        '#title' => $this->t('Search Analyser for @field', ['@field' => $field_label]),
        '#title_display' => 'invisible',
        '#options' => $analyser_options,
        '#default_value' => $current_config['search_analyser'] ?? '',
        '#empty_option' => $this->t('- Default -'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $global_search_analyser = $form_state->getValue('global_search_analyser', '');
    $field_analysers = $form_state->getValue('field_analysers', []);

    // Clean up empty values.
    $cleaned_analysers = [];
    foreach ($field_analysers as $field_id => $config) {
      if (!empty($config['index_analyser']) || !empty($config['search_analyser'])) {
        $cleaned_analysers[$field_id] = [
          'index_analyser' => $config['index_analyser'] ?? '',
          'search_analyser' => $config['search_analyser'] ?? '',
        ];
      }
    }

    $this->setConfiguration([
      'global_search_analyser' => $global_search_analyser,
      'field_analysers' => $cleaned_analysers,
    ]);
  }

  /**
   * Get available analyser options.
   *
   * @return array
   *   An array of analyser options keyed by analyser ID.
   */
  protected function getAnalyserOptions() {
    $options = [];

    try {
      $analysers = $this->entityTypeManager
        ->getStorage('es_analyser')
        ->loadMultiple();

      foreach ($analysers as $analyser_id => $analyser) {
        $options[$analyser_id] = $analyser->label();
      }
    }
    catch (\Exception $e) {
      \Drupal::logger('es_filter_analyser')->error('Error loading analysers: @message', ['@message' => $e->getMessage()]);
    }

    return $options;
  }

  /**
   * Get the human-readable label for a field type.
   *
   * @param string $field_type
   *   The field type machine name.
   *
   * @return string
   *   The field type label.
   */
  protected function getFieldTypeLabel($field_type) {
    try {
      $data_type_definition = $this->dataTypePluginManager->getDefinition($field_type);
      if (isset($data_type_definition['label'])) {
        return (string) $data_type_definition['label'];
      }
    }
    catch (\Exception $e) {
      // If we can't get the definition, fall back to the machine name.
    }

    // Fallback: return the machine name with first letter capitalized.
    return ucfirst(str_replace('_', ' ', $field_type));
  }

  /**
   * Check if a field type is a text field.
   *
   * @param string $field_type
   *   The field type.
   *
   * @return bool
   *   TRUE if the field is a text field, FALSE otherwise.
   */
  protected function isTextField($field_type) {
    $text_types = [
      'text',
      'string',
      'tokenized_text',
    ];

    return in_array($field_type, $text_types);
  }

  /**
   * Get the global search analyser.
   *
   * @return string|null
   *   The global search analyser ID or NULL if not configured.
   */
  public function getGlobalSearchAnalyser() {
    $configuration = $this->getConfiguration();
    return $configuration['global_search_analyser'] ?? NULL;
  }

  /**
   * Get the index analyser for a specific field.
   *
   * @param string $field_id
   *   The field ID.
   *
   * @return string|null
   *   The analyser ID or NULL if not configured.
   */
  public function getFieldIndexAnalyser($field_id) {
    $configuration = $this->getConfiguration();
    return $configuration['field_analysers'][$field_id]['index_analyser'] ?? NULL;
  }

  /**
   * Get the search analyser for a specific field.
   *
   * Falls back to global search analyser if no field-specific analyser is set.
   *
   * @param string $field_id
   *   The field ID.
   *
   * @return string|null
   *   The analyser ID or NULL if not configured.
   */
  public function getFieldSearchAnalyser($field_id) {
    $configuration = $this->getConfiguration();
    $field_analyser = $configuration['field_analysers'][$field_id]['search_analyser'] ?? NULL;

    // Return field-specific analyser if set, otherwise fall back to global.
    return $field_analyser ?: $this->getGlobalSearchAnalyser();
  }

  /**
   * Get all configured field analysers.
   *
   * @return array
   *   An array of field analyser configurations.
   */
  public function getFieldAnalysers() {
    $configuration = $this->getConfiguration();
    return $configuration['field_analysers'] ?? [];
  }

  /**
   * Get the analyser entity for a specific field and operation.
   *
   * @param string $field_id
   *   The field ID.
   * @param string $operation
   *   The operation ('index' or 'search').
   *
   * @return \Drupal\es_filter_analyser\Entity\AnalyserInterface|null
   *   The analyser entity or NULL if not found.
   */
  public function getFieldAnalyserEntity($field_id, $operation = 'index') {
    $analyser_id = $operation === 'search'
      ? $this->getFieldSearchAnalyser($field_id)
      : $this->getFieldIndexAnalyser($field_id);

    if (!$analyser_id) {
      return NULL;
    }

    try {
      return $this->entityTypeManager
        ->getStorage('es_analyser')
        ->load($analyser_id);
    }
    catch (\Exception $e) {
      \Drupal::logger('es_filter_analyser')->error('Error loading analyser @id: @message', [
        '@id' => $analyser_id,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
