<?php

namespace Drupal\es_filter_analyser\Plugin\FilterType;

use Drupal\es_filter_analyser\Plugin\FilterTypeBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides an N-Gram filter type.
 *
 * @FilterType(
 *   id = "ngram",
 *   label = @Translation("N-Gram"),
 *   description = @Translation("Generates n-grams of specified lengths from tokens.")
 * )
 */
class NGram extends FilterTypeBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'min_gram' => 1,
      'max_gram' => 2,
      'preserve_original' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, array $configuration = []) {
    $configuration += $this->defaultConfiguration();

    $form['min_gram'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum n-gram length'),
      '#default_value' => $configuration['min_gram'],
      '#description' => $this->t('Minimum length of characters in a gram.'),
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['max_gram'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum n-gram length'),
      '#default_value' => $configuration['max_gram'],
      '#description' => $this->t('Maximum length of characters in a gram.'),
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['preserve_original'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Preserve original'),
      '#default_value' => $configuration['preserve_original'],
      '#description' => $this->t('If checked, keeps the original token as well as the n-grams.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $min = $form_state->getValue('min_gram');
    $max = $form_state->getValue('max_gram');

    if ($min > $max) {
      $form_state->setErrorByName('max_gram', $this->t('Maximum n-gram length must be greater than or equal to minimum n-gram length.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array $values) {
    return [
      'min_gram' => (int) $values['min_gram'],
      'max_gram' => (int) $values['max_gram'],
      'preserve_original' => (bool) $values['preserve_original'],
    ];
  }

}
