<?php

namespace Drupal\es_filter_analyser\Plugin\FilterType;

use Drupal\es_filter_analyser\Plugin\FilterTypeBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a Shingle filter type.
 *
 * @FilterType(
 *   id = "shingle",
 *   label = @Translation("Shingle"),
 *   description = @Translation("Creates word n-grams (shingles) from tokens.")
 * )
 */
class Shingle extends FilterTypeBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'min_shingle_size' => 2,
      'max_shingle_size' => 2,
      'output_unigrams' => TRUE,
      'output_unigrams_if_no_shingles' => FALSE,
      'token_separator' => ' ',
      'filler_token' => '_',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, array $configuration = []) {
    $configuration += $this->defaultConfiguration();

    $form['min_shingle_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum shingle size'),
      '#default_value' => $configuration['min_shingle_size'],
      '#description' => $this->t('Minimum number of tokens per shingle.'),
      '#required' => TRUE,
      '#min' => 2,
    ];

    $form['max_shingle_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum shingle size'),
      '#default_value' => $configuration['max_shingle_size'],
      '#description' => $this->t('Maximum number of tokens per shingle.'),
      '#required' => TRUE,
      '#min' => 2,
    ];

    $form['output_unigrams'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Output unigrams'),
      '#default_value' => $configuration['output_unigrams'],
      '#description' => $this->t('If checked, outputs individual tokens as well as shingles.'),
    ];

    $form['output_unigrams_if_no_shingles'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Output unigrams if no shingles'),
      '#default_value' => $configuration['output_unigrams_if_no_shingles'],
      '#description' => $this->t('If checked, outputs unigrams when no shingles can be created.'),
    ];

    $form['token_separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token separator'),
      '#default_value' => $configuration['token_separator'],
      '#description' => $this->t('Character to use as separator between tokens in a shingle.'),
      '#maxlength' => 10,
    ];

    $form['filler_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filler token'),
      '#default_value' => $configuration['filler_token'],
      '#description' => $this->t('Token to use as filler for positions without a token.'),
      '#maxlength' => 10,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $min = $form_state->getValue('min_shingle_size');
    $max = $form_state->getValue('max_shingle_size');

    if ($min > $max) {
      $form_state->setErrorByName('max_shingle_size', $this->t('Maximum shingle size must be greater than or equal to minimum shingle size.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array $values) {
    return [
      'min_shingle_size' => (int) $values['min_shingle_size'],
      'max_shingle_size' => (int) $values['max_shingle_size'],
      'output_unigrams' => (bool) $values['output_unigrams'],
      'output_unigrams_if_no_shingles' => (bool) $values['output_unigrams_if_no_shingles'],
      'token_separator' => $values['token_separator'],
      'filler_token' => $values['filler_token'],
    ];
  }

}
