<?php

/**
 * @file
 * Functions to support Event Horizon theme settings.
 */

use Drupal\Core\Form\FormStateInterface;

/**
 * Implements hook_form_FORM_ID_alter() for system_theme_settings.
 */
function event_horizon_form_system_theme_settings_alter(&$form, FormStateInterface $form_state): void {
  $form['#attached']['library'][] = 'event_horizon/color-picker';

  $color_config['primary'] = [
    'colors' => [
      'base_primary_color' => 'Primary base color',
    ],
    'schemes' => [
      'default' => [
        'label' => 'Dark Pink',
        'colors' => [
          'base_primary_color' => '#e70077',
        ],
      ],
      'firehouse' => [
        'label' => 'Firehouse',
        'colors' => [
          'base_primary_color' => '#a30f0f',
        ],
      ],
      'ice' => [
        'label' => 'Ice',
        'colors' => [
          'base_primary_color' => '#57919e',
        ],
      ],
      'plum' => [
        'label' => 'Plum',
        'colors' => [
          'base_primary_color' => '#7a4587',
        ],
      ],
      'slate' => [
        'label' => 'Slate',
        'colors' => [
          'base_primary_color' => '#47625b',
        ],
      ],
    ],
  ];

  $color_config['secondary'] = [
    'colors' => [
      'base_secondary_color' => 'Secondary base color',
    ],
    'schemes' => [
      'default' => [
        'label' => 'Teal',
        'colors' => [
          'base_secondary_color' => '#93e7ff',
        ],
      ],
      'firehouse' => [
        'label' => 'Firehouse',
        'colors' => [
          'base_secondary_color' => '#a30f0f',
        ],
      ],
      'ice' => [
        'label' => 'Ice',
        'colors' => [
          'base_secondary_color' => '#57919e',
        ],
      ],
      'plum' => [
        'label' => 'Plum',
        'colors' => [
          'base_secondary_color' => '#7a4587',
        ],
      ],
      'slate' => [
        'label' => 'Slate',
        'colors' => [
          'base_secondary_color' => '#47625b',
        ],
      ],
    ],
  ];

  $form['#attached']['drupalSettings']['event_horizon']['colorSchemes']['primary'] = $color_config['primary']['schemes'];
  $form['#attached']['drupalSettings']['event_horizon']['colorSchemes']['secondary'] = $color_config['secondary']['schemes'];

  $form['event_horizon_settings']['event_horizon_utilities'] = [
    '#type' => 'fieldset',
    '#title' => t('Event Horizon Utilities'),
  ];
  $form['event_horizon_settings']['event_horizon_utilities']['event_horizon_color_scheme'] = [
    '#type' => 'fieldset',
    '#title' => t('Event Horizon Color Scheme Settings'),
  ];
  $form['event_horizon_settings']['event_horizon_utilities']['event_horizon_color_scheme']['description'] = [
    '#type' => 'html_tag',
    '#tag' => 'p',
    '#value' => t('These settings adjust the look and feel of the Event Horizon theme. Changing the color below will change the base hue, saturation, and lightness values the Event Horizon theme uses to determine its internal colors.'),
  ];

  $schemes = [
    'primary' => 'Primary',
    'secondary' => 'Secondary',
  ];

  $scheme_options = [
    'primary' => [
      'default' => t('Dark Pink (Default)'),
      'firehouse' => t('Firehouse'),
      'ice' => t('Ice'),
      'plum' => t('Plum'),
      'slate' => t('Slate'),
    ],
    'secondary' => [
      'default' => t('Teal (Default)'),
      'firehouse' => t('Firehouse'),
      'ice' => t('Ice'),
      'plum' => t('Plum'),
      'slate' => t('Slate'),
    ],
  ];

  foreach ($schemes as $scheme_id => $scheme_label) {
    $form['event_horizon_settings']['event_horizon_utilities']['event_horizon_color_scheme']['color_scheme_' . $scheme_id] = [
      '#type' => 'select',
      '#title' => t('Event Horizon @scheme Color', ['@scheme' => $scheme_label]),
      '#empty_option' => t('Custom'),
      '#empty_value' => '',
      '#options' => $scheme_options[$scheme_id],
      '#input' => FALSE,
      '#wrapper_attributes' => [
        'style' => 'display:none;',
      ],
    ];

    foreach ($color_config[$scheme_id]['colors'] as $key => $title) {
      $form['event_horizon_settings']['event_horizon_utilities']['event_horizon_color_scheme'][$key] = [
        '#type' => 'textfield',
        '#maxlength' => 7,
        '#size' => 10,
        '#title' => t($title),
        '#description' => t('Enter color in hexadecimal format (#abc123).') . '<br/>' . t('Derivatives will be formed from this color.'),
        '#config_target' => "event_horizon.settings:$key",
        '#attributes' => [
          // Regex copied from Color::validateHex()
          'pattern' => '^[#]?([0-9a-fA-F]{3}){1,2}$',
        ],
        '#wrapper_attributes' => [
          'data-drupal-selector' => 'eventhorizon-color-picker',
        ],
      ];
    }
  }

  // Graphic settings, only available when file.module is enabled.
  if (\Drupal::moduleHandler()->moduleExists('file')) {
    $form['graphic'] = [
      '#type' => 'details',
      '#title' => t('Home page graphic'),
      '#open' => TRUE,
    ];
    $form['graphic']['default_graphic'] = [
      '#type' => 'checkbox',
      '#title' => t('Use the graphic supplied by the theme'),
      '#config_target' => "event_horizon.settings:graphic.use_default",
      '#tree' => FALSE,
    ];
    $form['graphic']['settings'] = [
      '#type' => 'container',
      '#states' => [
        // Hide the graphic settings when using the default graphic.
        'invisible' => [
          'input[name="default_graphic"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['graphic']['settings']['graphic_path'] = [
      '#type' => 'textfield',
      '#title' => t('Path to custom graphic'),
      '#config_target' => "event_horizon.settings:graphic.path",
    ];
    $form['graphic']['settings']['graphic_upload'] = [
      '#type' => 'file',
      '#title' => t('Upload graphic image'),
      '#description' => t("If you don't have direct file access to the server, use this field to upload your graphic."),
      '#upload_validators' => [
        'FileExtension' => [
          'extensions' => 'png gif jpg jpeg apng webp svg',
        ],
      ],
    ];
    $form['texture'] = [
      '#type' => 'details',
      '#title' => t('Home page texture'),
      '#open' => TRUE,
    ];
    $form['texture']['default_texture'] = [
      '#type' => 'checkbox',
      '#title' => t('Use the texture supplied by the theme'),
      '#config_target' => "event_horizon.settings:texture.use_default",
      '#tree' => FALSE,
    ];
    $form['texture']['settings'] = [
      '#type' => 'container',
      '#states' => [
        // Hide the texture settings when using the default texture.
        'invisible' => [
          'input[name="default_texture"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['texture']['settings']['texture_path'] = [
      '#type' => 'textfield',
      '#title' => t('Path to custom texture'),
      '#config_target' => "event_horizon.settings:texture.path",
    ];
    $form['texture']['settings']['texture_upload'] = [
      '#type' => 'file',
      '#title' => t('Upload texture image'),
      '#description' => t("If you don't have direct file access to the server, use this field to upload your graphic."),
      '#upload_validators' => [
        'FileExtension' => [
          'extensions' => 'png gif jpg jpeg apng webp avif',
        ],
      ],
    ];
    $form['video'] = [
      '#type' => 'details',
      '#title' => t('Home page video'),
      '#open' => TRUE,
    ];
    $form['video']['default_video'] = [
      '#type' => 'checkbox',
      '#title' => t('Use the video supplied by the theme'),
      '#config_target' => "event_horizon.settings:video.use_default",
      '#tree' => FALSE,
    ];
    $form['video']['settings'] = [
      '#type' => 'container',
      '#states' => [
        // Hide the video settings when using the default video.
        'invisible' => [
          'input[name="default_video"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['video']['settings']['video_path'] = [
      '#type' => 'textfield',
      '#title' => t('Path to custom video'),
      '#config_target' => "event_horizon.settings:video.path",
    ];
    $form['video']['settings']['video_upload'] = [
      '#type' => 'file',
      '#title' => t('Upload video file'),
      '#description' => t("If you don't have direct file access to the server, use this field to upload your video file."),
      '#upload_validators' => [
        'FileExtension' => [
          'extensions' => 'mp4 webm mpg mpeg',
        ],
      ],
    ];
    $form['video']['settings']['video_poster_path'] = [
      '#type' => 'textfield',
      '#title' => t('Path to custom video poster image'),
      '#config_target' => "event_horizon.settings:video_poster.path",
    ];
    $form['video']['settings']['video_poster_upload'] = [
      '#type' => 'file',
      '#title' => t('Upload video poster image'),
      '#description' => t("If you don't have direct file access to the server, use this field to upload your video poster image."),
      '#upload_validators' => [
        'FileExtension' => [
          'extensions' => 'webp png jpeg jpg avif',
        ],
      ],
    ];
    $form['#validate'][] = '_event_horizon_form_system_theme_settings_alter_validate';
    $form['#submit'][] = '_event_horizon_form_system_theme_settings_alter_submit';
  }

}

/**
 * Cleanup settings.
 */
function _event_horizon_form_system_theme_settings_alter_validate(&$form, FormStateInterface $form_state) {
  if (\Drupal::moduleHandler()->moduleExists('file')) {
    // Check for a new uploaded graphic.
    if (isset($form['graphic'])) {
      $file = _file_save_upload_from_form($form['graphic']['settings']['graphic_upload'], $form_state, 0);
      if ($file) {
        // Put the temporary file in form_values so we can save it on submit.
        $form_state->setValue('graphic_upload', $file);
      }
    }

    // Check for a new uploaded texture.
    if (isset($form['texture'])) {
      $file = _file_save_upload_from_form($form['texture']['settings']['texture_upload'], $form_state, 0);
      if ($file) {
        // Put the temporary file in form_values so we can save it on submit.
        $form_state->setValue('texture_upload', $file);
      }
    }

    // Check for a new uploaded video.
    if (isset($form['video']['settings']['video_upload'])) {
      $file = _file_save_upload_from_form($form['video']['settings']['video_upload'], $form_state, 0);
      if ($file) {
        // Put the temporary file in form_values so we can save it on submit.
        $form_state->setValue('video_upload', $file);
      }
    }

    // Check for a new uploaded video poster.
    if (isset($form['video']['settings']['video_poster_upload'])) {
      $file = _file_save_upload_from_form($form['video']['settings']['video_poster_upload'], $form_state, 0);
      if ($file) {
        // Put the temporary file in form_values so we can save it on submit.
        $form_state->setValue('video_poster_upload', $file);
      }
    }
  }

  // When intending to use the default graphic, unset the graphic_path.
  if ($form_state->getValue('default_graphic')) {
    $form_state->unsetValue('graphic_path');
  }

  // When intending to use the default texture, unset the texture_path.
  if ($form_state->getValue('default_texture')) {
    $form_state->unsetValue('texture_path');
  }

  // When intending to use the default video, unset the paths.
  if ($form_state->getValue('default_video')) {
    $form_state->unsetValue('video_path');
    $form_state->unsetValue('video_poster_path');
  }
}

/**
 * Cleanup settings.
 */
function _event_horizon_form_system_theme_settings_alter_submit(&$form, FormStateInterface $form_state) {
  $values = $form_state->getValues();

  // If the user uploaded a new graphic, save it to a permanent location
  // and use it in place of the default theme-provided file.
  $default_scheme = \Drupal::config('system.file')->get('default_scheme');
  $file_system = \Drupal::service('file_system');

  try {
    if (!empty($values['graphic_upload'])) {
      $graphic_uri = $values['graphic_upload']->getFileUri();
      $filename = $file_system->copy($graphic_uri, $default_scheme . '://');
      $form_state->setValue('graphic_path', $filename);
    }
  }
  catch (FileException) {
    // Ignore.
  }
  $form_state->unsetValue('graphic_upload');

  try {
    if (!empty($values['texture_upload'])) {
      $texture_uri = $values['texture_upload']->getFileUri();
      $filename = $file_system->copy($texture_uri, $default_scheme . '://');
      $form_state->setValue('texture_path', $filename);
    }
  }
  catch (FileException) {
    // Ignore.
  }
  $form_state->unsetValue('texture_upload');

  try {
    if (!empty($values['video_upload'])) {
      $video_uri = $values['video_upload']->getFileUri();
      $filename = $file_system->copy($video_uri, $default_scheme . '://');
      $form_state->setValue('video_path', $filename);
    }
  }
  catch (FileException) {
    // Ignore.
  }
  $form_state->unsetValue('video_upload');

  try {
    if (!empty($values['video_poster_upload'])) {
      $video_poster_uri = $values['video_poster_upload']->getFileUri();
      $filename = $file_system->copy($video_poster_uri, $default_scheme . '://');
      $form_state->setValue('video_poster_path', $filename);
    }
  }
  catch (FileException) {
    // Ignore.
  }
  $form_state->unsetValue('video_poster_upload');
}
