<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Repository;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\node\NodeInterface;

/**
 * Repository for event-related database operations.
 */
final class EventRepository implements EventRepositoryInterface {

  /**
   * The entity type manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   */
  private ModuleHandlerInterface $moduleHandler;

  /**
   * Constructs a new EventRepository.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandlerInterface $moduleHandler,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public function getSessionsForTimeSlot(string $timeSlotId, array $contentTypes = ['session', 'special_session']): array {
    $sessions = [];

    foreach ($contentTypes as $contentType) {
      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->accessCheck()
        ->condition('type', $contentType)
        ->condition('status', 1)
        ->condition('field_time_slot', $timeSlotId)
        ->execute();

      if (!empty($query)) {
        /** @var \Drupal\node\NodeInterface[] $nodes */
        $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($query);
        $sessions = array_merge($sessions, $nodes);
      }
    }

    return $sessions;
  }

  /**
   * {@inheritdoc}
   */
  public function searchSessions(array $filters = [], int $limit = 50, int $offset = 0): array {
    $query = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck()
      ->condition('type', ['session', 'special_session'], 'IN')
      ->condition('status', 1);

    // Apply text search.
    if (!empty($filters['query'])) {
      $orGroup = $query->orConditionGroup()
        ->condition('title', '%' . $filters['query'] . '%', 'LIKE')
        ->condition('field_description', '%' . $filters['query'] . '%', 'LIKE');

      $query->condition($orGroup);
    }

    // Apply other filters.
    if (!empty($filters['category'])) {
      $query->condition('field_session_category', $filters['category']);
    }

    if (!empty($filters['room'])) {
      $query->condition('field_r', $filters['room']);
    }

    if (!empty($filters['speaker'])) {
      $query->condition('field_speakers', $filters['speaker']);
    }

    if (!empty($filters['audience_level'])) {
      $query->condition('field_audience', $filters['audience_level']);
    }

    if (isset($filters['is_training'])) {
      $query->condition('field_is_training', $filters['is_training']);
    }

    if (!empty($filters['time_slots'])) {
      $query->condition('field_time_slot', $filters['time_slots'], 'IN');
    }

    // Apply date range filtering if specified.
    if (!empty($filters['dates'])) {
      $this->addDateRangeConditions($query, $filters['dates']);
    }

    $query->range($offset, $limit);
    $query->sort('title');

    $nids = $query->execute();

    if (empty($nids)) {
      return [];
    }

    /** @var \Drupal\node\NodeInterface[] $nodes */
    return $this->entityTypeManager->getStorage('node')->loadMultiple($nids);
  }

  /**
   * {@inheritdoc}
   */
  public function getSessionById(string $sessionId, array $contentTypes = ['session', 'special_session']): ?NodeInterface {
    $nodes = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck()
      ->condition('nid', $sessionId)
      ->condition('type', $contentTypes, 'IN')
      ->condition('status', 1)
      ->execute();

    if (empty($nodes)) {
      return NULL;
    }

    /** @var \Drupal\node\NodeInterface $node */
    return $this->entityTypeManager->getStorage('node')->load(reset($nodes));
  }

  /**
   * {@inheritdoc}
   */
  public function getAllSpeakers(): array {
    $query = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck()
      ->condition('type', ['session', 'special_session'], 'IN')
      ->condition('status', 1)
      ->exists('field_speakers')
      ->execute();

    $speakers = [];
    if (!empty($query)) {
      /** @var \Drupal\node\NodeInterface[] $nodes */
      $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($query);
      foreach ($nodes as $node) {
        if ($node->hasField('field_speakers') && !$node->get('field_speakers')->isEmpty()) {
          foreach ($node->get('field_speakers') as $speaker) {
            /** @var \Drupal\user\UserInterface|null $speakerEntity */
            $speakerEntity = $speaker->entity ?? NULL;
            if ($speakerEntity) {
              $speakerName = $speakerEntity->getDisplayName();
              if (!in_array($speakerName, $speakers, TRUE)) {
                $speakers[] = $speakerName;
              }
            }
          }
        }
      }
    }

    return $speakers;
  }

  /**
   * Add date range conditions to the query.
   *
   * @param \Drupal\Core\Entity\Query\QueryInterface $query
   *   The entity query.
   * @param array $dateRange
   *   The date range arguments.
   */
  private function addDateRangeConditions($query, array $dateRange): void {
    $startDate = !empty($dateRange['start']) ? $this->normalizeDateTime($dateRange['start']) : NULL;
    $endDate = !empty($dateRange['end']) ? $this->normalizeDateTime($dateRange['end']) : NULL;

    if (!$startDate && !$endDate) {
      return;
    }

    // Get time slots that match the date range.
    $timeSlotIds = $this->getTimeSlotsInDateRange($startDate, $endDate);

    if (!empty($timeSlotIds)) {
      $query->condition('field_time_slot', $timeSlotIds, 'IN');
    }
    else {
      // If no time slots match, ensure no results are returned.
      $query->condition('nid', 0);
    }
  }

  /**
   * Get time slots that overlap with the specified date range.
   *
   * @param string|null $startDate
   *   Start date for filtering (Y-m-d H:i:s format).
   * @param string|null $endDate
   *   End date for filtering (Y-m-d H:i:s format).
   *
   * @return array
   *   Array of time slot IDs that overlap with the date range.
   */
  private function getTimeSlotsInDateRange(?string $startDate, ?string $endDate): array {
    try {
      /** @var \Drupal\taxonomy\TermInterface[] $timeSlots */
      $timeSlots = $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => 'time_slot']);

      $matchingTimeSlots = [];
      foreach ($timeSlots as $timeSlot) {
        if ($timeSlot->hasField('field_when') && !$timeSlot->get('field_when')->isEmpty()) {
          /** @var \Drupal\smart_date\Plugin\Field\FieldType\SmartDateItem $whenField */
          $whenField = $timeSlot->get('field_when')->first();
          if (!$whenField->isEmpty()) {
            $when_field_value = $whenField->getValue();
            $slotStart = date('Y-m-d H:i:s', (int) $when_field_value['value']);
            $slotEnd = $when_field_value['end_value'] ? date('Y-m-d H:i:s', (int) $when_field_value['end_value']) : $slotStart;

            // Check if time slot overlaps with the requested date range.
            $overlaps = TRUE;

            if ($startDate && $slotEnd < $startDate) {
              $overlaps = FALSE;
            }

            if ($endDate && $slotStart > $endDate) {
              $overlaps = FALSE;
            }

            if ($overlaps) {
              $matchingTimeSlots[] = $timeSlot->id();
            }
          }
        }
      }

      return $matchingTimeSlots;
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * Normalize date/time string to include time if not present.
   *
   * @param string $dateTime
   *   The date/time string to normalize.
   *
   * @return string
   *   Normalized date/time string.
   */
  private function normalizeDateTime(string $dateTime): string {
    // If it's just a date (YYYY-MM-DD), add default time.
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateTime)) {
      return $dateTime . ' 00:00:00';
    }

    // If it's date with time (YYYY-MM-DD HH:MM), add seconds.
    if (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/', $dateTime)) {
      return $dateTime . ':00';
    }

    return $dateTime;
  }

}
