<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Service;

use Drupal\event_platform_mcp\Repository\ConfigRepositoryInterface;
use Drupal\event_platform_mcp\Repository\EventRepositoryInterface;
use Drupal\event_platform_mcp\Repository\TaxonomyRepositoryInterface;
use Drupal\mcp\ServerFeatures\Resource;
use Drupal\node\NodeInterface;

/**
 * Unified service for event platform business logic.
 */
final class EventPlatformService implements EventPlatformServiceInterface {

  /**
   * The config repository.
   */
  private ConfigRepositoryInterface $configRepository;

  /**
   * The event repository.
   */
  private EventRepositoryInterface $eventRepository;

  /**
   * The taxonomy repository.
   */
  private TaxonomyRepositoryInterface $taxonomyRepository;

  /**
   * The date service.
   */
  private DateServiceInterface $dateService;

  /**
   * Constructs a new EventPlatformService.
   */
  public function __construct(
    ConfigRepositoryInterface $configRepository,
    EventRepositoryInterface $eventRepository,
    TaxonomyRepositoryInterface $taxonomyRepository,
    DateServiceInterface $dateService,
  ) {
    $this->configRepository = $configRepository;
    $this->eventRepository = $eventRepository;
    $this->taxonomyRepository = $taxonomyRepository;
    $this->dateService = $dateService;
  }

  /**
   * {@inheritdoc}
   */
  public function getEventOverview(array $arguments = []): array {
    try {
      $eventData = $this->configRepository->getEventDetails();

      if (!$eventData) {
        return [
          [
            'type' => 'error',
            'message' => 'Event details configuration not found',
          ],
        ];
      }

      return [
        [
          'type' => 'resource',
          'resource' => new Resource(
            uri: 'event/overview',
            name: 'Event Overview',
            description: 'General information about the event',
            mimeType: 'application/json',
            text: json_encode($eventData, JSON_UNESCAPED_UNICODE),
          ),
        ],
      ];
    }
    catch (\Exception $e) {
      return [
        [
          'type' => 'error',
          'message' => 'Error retrieving event overview: ' . $e->getMessage(),
        ],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSchedule(array $arguments = []): array {
    try {
      $date = $arguments['date'] ?? NULL;
      $includeDetails = $arguments['include_details'] ?? FALSE;

      $timeSlots = $this->taxonomyRepository->getAllTimeSlots();

      if (empty($timeSlots)) {
        return [
          [
            'type' => 'error',
            'message' => 'No time slots found',
          ],
        ];
      }

      $schedule = [];
      foreach ($timeSlots as $timeSlot) {
        $timeSlotData = [
          'id' => $timeSlot->id(),
          'name' => $timeSlot->getName(),
          'when' => NULL,
          'sessions' => [],
        ];

        // Get time information.
        if ($timeSlot->hasField('field_when') && !$timeSlot->get('field_when')->isEmpty()) {
          /** @var \Drupal\smart_date\Plugin\Field\FieldType\SmartDateItem $whenField */
          $whenField = $timeSlot->get('field_when')->first();
          if (!$whenField->isEmpty()) {
            $when_field_value = $whenField->getValue();
            $timeSlotData['when'] = $this->dateService->formatTimeSlotData($when_field_value);
          }
        }

        // Filter by date if specified.
        if (!$this->dateService->filterByDate($timeSlotData, $date)) {
          continue;
        }

        // Get sessions for this time slot.
        $sessions = $this->getSessionsForTimeSlot($timeSlot->id(), $includeDetails);
        $timeSlotData['sessions'] = $sessions;

        $schedule[] = $timeSlotData;
      }

      // Sort by start time.
      $schedule = $this->dateService->sortByStartTime($schedule);

      return [
        [
          'type' => 'resource',
          'resource' => new Resource(
            uri: 'event/schedule',
            name: 'Event Schedule',
            description: 'Complete event schedule with all sessions organized by time',
            mimeType: 'application/json',
            text: json_encode($schedule, JSON_UNESCAPED_UNICODE),
          ),
        ],
      ];
    }
    catch (\Exception $e) {
      return [
        [
          'type' => 'error',
          'message' => 'Error retrieving schedule: ' . $e->getMessage(),
        ],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function searchSessions(array $arguments): array {
    $errors = [];
    $filters = [];

    // Process text search.
    if (!empty($arguments['query'])) {
      $filters['query'] = $arguments['query'];
    }

    // Process other filters.
    if (!empty($arguments['filters'])) {
      $filterArgs = $arguments['filters'];

      // Category filter.
      if (!empty($filterArgs['category'])) {
        $categoryId = $this->taxonomyRepository->findTermIdByName('session_category', $filterArgs['category']);
        if ($categoryId) {
          $filters['category'] = $categoryId;
        }
        else {
          $errors[] = "Category '{$filterArgs['category']}' not found";
        }
      }

      // Room filter.
      if (!empty($filterArgs['room'])) {
        $roomId = $this->taxonomyRepository->findTermIdByName('room', $filterArgs['room']);
        if ($roomId) {
          $filters['room'] = $roomId;
        }
        else {
          $errors[] = "Room '{$filterArgs['room']}' not found";
        }
      }

      // Speaker filter.
      if (!empty($filterArgs['speaker'])) {
        $speakerId = $this->configRepository->findUserIdByName($filterArgs['speaker']);
        if ($speakerId) {
          $filters['speaker'] = $speakerId;
        }
        else {
          $errors[] = "Speaker '{$filterArgs['speaker']}' not found";
        }
      }

      // Audience level filter.
      if (!empty($filterArgs['audience_level'])) {
        $audienceId = $this->taxonomyRepository->findTermIdByName('session_audience', $filterArgs['audience_level']);
        if ($audienceId) {
          $filters['audience_level'] = $audienceId;
        }
        else {
          $errors[] = "Audience level '{$filterArgs['audience_level']}' not found";
        }
      }

      // Training filter.
      if (isset($filterArgs['is_training'])) {
        $filters['is_training'] = $filterArgs['is_training'];
      }

      // Date range filter.
      if (!empty($filterArgs['dates'])) {
        $dateRangeArgs = $filterArgs['dates'];
        $eventRange = $this->getEventDateRange();
        $validationResult = $this->dateService->validateDateRange($dateRangeArgs, $eventRange);

        if (!$validationResult['valid']) {
          $errors = array_merge($errors, $validationResult['errors']);
        }
        else {
          // Store date range for direct filtering in repository.
          $filters['dates'] = $dateRangeArgs;
        }
      }
    }

    // If there are validation errors, return them.
    if (!empty($errors)) {
      return [
        [
          'type' => 'error',
          'message' => 'Validation errors: ' . implode(', ', $errors),
        ],
      ];
    }

    // Execute search.
    $limit = $arguments['limit'] ?? 50;
    $includeDetails = $arguments['include_details'] ?? TRUE;
    $sessions = $this->eventRepository->searchSessions($filters, $limit);

    $resources = [];
    foreach ($sessions as $session) {
      $sessionData = $this->formatSessionData($session, $includeDetails);
      $resources[] = [
        "type" => "resource",
        "resource" => new Resource(
          uri: "session/{$session->id()}",
          name: $session->getTitle(),
          description: NULL,
          mimeType: 'text/plain',
          text: (string) json_encode([$sessionData], JSON_UNESCAPED_UNICODE),
        ),
      ];
    }

    return $resources;
  }

  /**
   * {@inheritdoc}
   */
  public function formatSessionData(NodeInterface $node, bool $includeDetails = FALSE): array {
    $sessionData = [
      'title' => $node->getTitle(),
      'type' => $node->bundle(),
      'url' => $node->toUrl()->setAbsolute()->toString(),
    ];

    if ($includeDetails) {
      $sessionData['description'] = $node->hasField('field_description') ?
        $node->get('field_description')->getString() : NULL;
      $sessionData['is_training'] = $node->hasField('field_is_training') && $node->get('field_is_training')->getValue();
      $sessionData['is_non_session'] = $node->hasField('field_is_non_session') && $node->get('field_is_non_session')->getValue();

      // Get speakers.
      if ($node->hasField('field_speakers') && !$node->get('field_speakers')->isEmpty()) {
        $speakers = [];
        foreach ($node->get('field_speakers') as $speaker) {
          /** @var \Drupal\user\UserInterface $speaker_entity */
          $speaker_entity = $speaker->entity;
          if ($speaker_entity) {
            $speakers[] = [
              'name' => $speaker_entity->get('field_display_name')->getString(),
              'biography' => $speaker_entity->get('field_bio')->getString(),
            ];
          }
        }
        $sessionData['speakers'] = $speakers;
      }

      // Get room.
      if ($node->hasField('field_r') && !$node->get('field_r')->isEmpty()) {
        $roomEntity = $node->get('field_r')->entity;
        if ($roomEntity) {
          $sessionData['room'] = [
            'id' => $roomEntity->id(),
            'name' => $roomEntity->getName(),
          ];
        }
      }

      // Get categories.
      if ($node->hasField('field_session_category') && !$node->get('field_session_category')->isEmpty()) {
        $categories = [];
        foreach ($node->get('field_session_category') as $category) {
          $categoryEntity = $category->entity;
          if ($categoryEntity) {
            $categories[] = [
              'id' => $categoryEntity->id(),
              'name' => $categoryEntity->getName(),
            ];
          }
        }
        $sessionData['categories'] = $categories;
      }

      // Get audience.
      if ($node->hasField('field_audience') && !$node->get('field_audience')->isEmpty()) {
        $audience = [];
        foreach ($node->get('field_audience') as $audienceItem) {
          $audienceEntity = $audienceItem->entity;
          if ($audienceEntity) {
            $audience[] = [
              'id' => $audienceEntity->id(),
              'name' => $audienceEntity->getName(),
            ];
          }
        }
        $sessionData['audience'] = $audience;
      }

      // Get language.
      if ($node->hasField('field_language') && !$node->get('field_language')->isEmpty()) {
        $language = [];
        foreach ($node->get('field_language') as $languageItem) {
          $languageEntity = $languageItem->entity;
          if ($languageEntity) {
            $language[] = [
              'id' => $languageEntity->id(),
              'name' => $languageEntity->getName(),
            ];
          }
        }
        $sessionData['language'] = $language;
      }
    }

    return $sessionData;
  }

  /**
   * {@inheritdoc}
   */
  public function getSessionsForTimeSlot(string $timeSlotId, bool $includeDetails = FALSE): array {
    $sessions = $this->eventRepository->getSessionsForTimeSlot($timeSlotId);
    $formattedSessions = [];

    foreach ($sessions as $session) {
      $formattedSessions[] = $this->formatSessionData($session, $includeDetails);
    }

    return $formattedSessions;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableCategories(): array {
    $terms = $this->taxonomyRepository->getTermsByVocabulary('session_category');
    $categories = [];
    foreach ($terms as $term) {
      $categories[] = $term->getName();
    }
    return $categories;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableRooms(): array {
    $terms = $this->taxonomyRepository->getTermsByVocabulary('room');
    $rooms = [];
    foreach ($terms as $term) {
      $rooms[] = $term->getName();
    }
    return $rooms;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableSpeakers(): array {
    return $this->eventRepository->getAllSpeakers();
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableAudienceLevels(): array {
    $terms = $this->taxonomyRepository->getTermsByVocabulary('session_audience');
    $audienceLevels = [];
    foreach ($terms as $term) {
      $audienceLevels[] = $term->getName();
    }
    return $audienceLevels;
  }

  /**
   * {@inheritdoc}
   */
  public function getEventDateRange(): ?array {
    $eventRange = $this->configRepository->getEventDateRange();
    if (!$eventRange) {
      return NULL;
    }

    return [
      'min' => $eventRange['start'],
      'max' => $eventRange['end'],
    ];
  }

}
