<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Service;

use Drupal\node\NodeInterface;

/**
 * Interface for unified event platform business logic.
 */
interface EventPlatformServiceInterface {

  /**
   * Get event overview information.
   *
   * @param array $arguments
   *   Optional arguments for the request.
   *
   * @return array
   *   Array containing event overview data or error information.
   */
  public function getEventOverview(array $arguments = []): array;

  /**
   * Get event schedule.
   *
   * @param array $arguments
   *   Optional arguments including 'date' and 'include_details'.
   *
   * @return array
   *   Array containing schedule data or error information.
   */
  public function getSchedule(array $arguments = []): array;

  /**
   * Search sessions with filters.
   *
   * @param array $arguments
   *   Search arguments including query, filters, limit, and include_details.
   *
   * @return array
   *   Array containing search results or error information.
   */
  public function searchSessions(array $arguments): array;

  /**
   * Format session data from a node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The session node.
   * @param bool $includeDetails
   *   Whether to include detailed information.
   *
   * @return array
   *   Formatted session data.
   */
  public function formatSessionData(NodeInterface $node, bool $includeDetails = FALSE): array;

  /**
   * Get sessions for a specific time slot.
   *
   * @param string $timeSlotId
   *   The time slot ID.
   * @param bool $includeDetails
   *   Whether to include detailed information.
   *
   * @return array
   *   Array of formatted session data.
   */
  public function getSessionsForTimeSlot(string $timeSlotId, bool $includeDetails = FALSE): array;

  /**
   * Get available session categories.
   *
   * @return array
   *   Array of category names.
   */
  public function getAvailableCategories(): array;

  /**
   * Get available rooms.
   *
   * @return array
   *   Array of room names.
   */
  public function getAvailableRooms(): array;

  /**
   * Get available speakers.
   *
   * @return array
   *   Array of speaker names.
   */
  public function getAvailableSpeakers(): array;

  /**
   * Get available audience levels.
   *
   * @return array
   *   Array of audience level names.
   */
  public function getAvailableAudienceLevels(): array;

  /**
   * Get event date range.
   *
   * @return array|null
   *   Array with 'min' and 'max' dates or NULL if not configured.
   */
  public function getEventDateRange(): ?array;

}
