<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Plugin\Mcp;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\event_platform_mcp\Service\EventPlatformServiceInterface;
use Drupal\mcp\Attribute\Mcp;
use Drupal\mcp\Plugin\McpPluginBase;
use Drupal\mcp\ServerFeatures\Resource;
use Drupal\mcp\ServerFeatures\ResourceTemplate;
use Drupal\mcp\ServerFeatures\Tool;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * MCP plugin for the content created by Event Platform.
 */
#[Mcp(
  id: 'event-platform',
  name: new TranslatableMarkup('Event Platform Content'),
  description: new TranslatableMarkup(
    'Provides comprehensive access to event information, sessions, schedule, and venue details.'
  ),
)]
final class EventPlatformContent extends McpPluginBase {

  /**
   * The module handler.
   */
  private ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The event platform service.
   */
  private EventPlatformServiceInterface $eventPlatformService;

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): EventPlatformContent {
    $instance = parent::create(
      $container,
      $configuration,
      $plugin_id,
      $plugin_definition,
    );

    $instance->moduleHandler = $container->get('module_handler');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->eventPlatformService = $container->get('event_platform_mcp.event_platform_service');

    return $instance;
  }

  /**
   * {@inheritDoc}
   */
  public function checkRequirements(): bool {
    return $this->moduleHandler->moduleExists('node') &&
           $this->moduleHandler->moduleExists('taxonomy') &&
           $this->moduleHandler->moduleExists('config_pages');
  }

  /**
   * {@inheritDoc}
   */
  public function getRequirementsDescription(): string {
    if (!$this->checkRequirements()) {
      return $this->t('The Node, Taxonomy, and Config Pages modules must be enabled to use this plugin.')->render();
    }
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    $config = parent::defaultConfiguration();
    $config['config']['enabled_content_types'] = ['session'];
    $config['config']['enabled_vocabularies'] = ['time_slot', 'room', 'session_category', 'session_audience'];
    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $config = $this->getConfiguration();
    /** @var \Drupal\node\NodeTypeInterface[] $nodeTypes */
    $nodeTypes = $this->entityTypeManager->getStorage('node_type')
      ->loadMultiple();

    $form['enabled_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled content types'),
      '#description' => $this->t('Select which content types should be available through MCP.'),
      '#options' => array_map(
        static fn($nodeType) => $nodeType->label(),
        $nodeTypes
      ),
      '#default_value' => $config['config']['enabled_content_types'] ?? [],
    ];

    $form['enabled_vocabularies'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled taxonomies'),
      '#description' => $this->t('Select which taxonomy vocabularies should be available.'),
      '#options' => [
        'time_slot' => $this->t('Time slot'),
        'room' => $this->t('Room'),
        'session_category' => $this->t('Session category'),
        'session_audience' => $this->t('Session audience'),
        'language' => $this->t('Language'),
      ],
      '#default_value' => $config['config']['enabled_vocabularies'] ?? [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getResources(): array {
    $resources = [];

    $resources[] = new Resource(
      uri: 'event/overview',
      name: 'Event Overview',
      description: 'General information about the event',
      mimeType: 'application/json',
      text: NULL,
    );

    $resources[] = new Resource(
      uri: 'event/schedule',
      name: 'Event Schedule',
      description: 'Complete event schedule with all sessions organized by time',
      mimeType: 'application/json',
      text: NULL,
    );

    return $resources;
  }

  /**
   * {@inheritdoc}
   */
  public function getResourceTemplates(): array {
    $resourceTemplates = [];

    $resourceTemplates[] = new ResourceTemplate(
      uriTemplate: "session/{id}",
      name: 'Session detail',
      description: "Detailed information about a specific session",
      mimeType: 'application/json',
    );

    return $resourceTemplates;
  }

  /**
   * {@inheritdoc}
   */
  public function getTools(): array {
    return [
      new Tool(
        name: 'get_event_overview',
        description: 'Get general information about the event including dates, location, and status.',
        inputSchema: [
          'type' => 'object',
          'properties' => [
            'include_details' => [
              'type' => 'boolean',
              'description' => 'Whether to include detailed event information',
              'default' => TRUE,
            ],
          ],
          'required' => [],
        ],
      ),
      new Tool(
        name: 'get_schedule',
        description: 'Get the complete event schedule with all sessions organized by day and time.',
        inputSchema: [
          'type' => 'object',
          'properties' => [
            'date' => [
              'type' => 'string',
              'description' => 'Specific date to get schedule for (YYYY-MM-DD format)',
              'format' => 'date',
            ],
            'include_details' => [
              'type' => 'boolean',
              'description' => 'Whether to include detailed session information',
              'default' => FALSE,
            ],
          ],
          'required' => [],
        ],
      ),
      new Tool(
        name: 'search_sessions',
        description: 'Advanced search for sessions with multiple filters. Date filters must be within the event date range.',
        inputSchema: [
          'type' => 'object',
          'properties' => [
            'query' => [
              'type' => 'string',
              'description' => 'Text to search for in session titles and descriptions',
            ],
            'filters' => [
              'type' => 'object',
              'description' => 'Additional filters to apply',
              'properties' => [
                'category' => [
                  'type' => 'string',
                  'description' => 'Category name to filter by',
                  'enum' => $this->eventPlatformService->getAvailableCategories(),
                ],
                'room' => [
                  'type' => 'string',
                  'description' => 'Room name to filter by',
                  'enum' => $this->eventPlatformService->getAvailableRooms(),
                ],
                'speaker' => [
                  'type' => 'string',
                  'description' => 'Speaker name to filter by',
                  'enum' => $this->eventPlatformService->getAvailableSpeakers(),
                ],
                'audience_level' => [
                  'type' => 'string',
                  'description' => 'Audience level to filter by',
                  'enum' => $this->eventPlatformService->getAvailableAudienceLevels(),
                ],
                'is_training' => [
                  'type' => 'boolean',
                  'description' => 'Whether to include only training sessions',
                ],
                'dates' => [
                  'type' => 'object',
                  'description' => 'Date and time range to filter by. All dates must be within the event date range. To search for sessions at a specific time, use start and end with the same date and time range (e.g., start: "2024-03-15 12:00", end: "2024-03-15 12:59").',
                  'properties' => [
                    'start' => [
                      'type' => 'string',
                      'description' => 'Start date and time (YYYY-MM-DD or YYYY-MM-DD HH:MM format). Must be within event date range.',
                      'pattern' => '^(\d{4}-\d{2}-\d{2})(\s\d{2}:\d{2})?$',
                    ],
                    'end' => [
                      'type' => 'string',
                      'description' => 'End date and time (YYYY-MM-DD or YYYY-MM-DD HH:MM format). Must be within event date range.',
                      'pattern' => '^(\d{4}-\d{2}-\d{2})(\s\d{2}:\d{2})?$',
                    ],
                  ],
                  'required' => [],
                ],
              ],
            ],
            'limit' => [
              'type' => 'integer',
              'description' => 'Maximum number of results to return',
              'default' => 50,
            ],
            'include_details' => [
              'type' => 'boolean',
              'description' => 'Whether to include detailed session information',
              'default' => TRUE,
            ],
          ],
          'required' => [],
        ],
      ),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function executeTool(string $toolId, mixed $arguments): array {
    return match ($toolId) {
      'get_event_overview' => $this->eventPlatformService->getEventOverview($arguments),
      'get_schedule' => $this->eventPlatformService->getSchedule($arguments),
      'search_sessions' => $this->eventPlatformService->searchSessions($arguments),
      default => throw new \InvalidArgumentException("Unknown tool: $toolId"),
    };
  }

  /**
   * {@inheritdoc}
   */
  public function readResource(string $resourceId): array {
    $parts = explode('/', $resourceId);

    if (count($parts) < 2) {
      throw new \InvalidArgumentException("Invalid resource ID format: $resourceId");
    }

    if ($parts[0] !== 'event' && $parts[0] !== 'session') {
      throw new \InvalidArgumentException("Unknown resource type: {$parts[0]}");
    }

    return match ($parts[0]) {
      'event' => match ($parts[1]) {
        'overview' => $this->eventPlatformService->getEventOverview([]),
        'schedule' => $this->eventPlatformService->getSchedule([]),
        default => throw new \InvalidArgumentException("Unknown event resource: $parts[1]"),
      },
      'sessions' => match ($parts[1]) {
        'search' => $this->eventPlatformService->searchSessions([]),
        default => throw new \InvalidArgumentException("Unknown sessions resource: $parts[2]"),
      },
      'session' => match (count($parts)) {
        2 => $this->readContentItem(array_unique(array_filter($this->getConfiguration()['config']['enabled_content_types'])), $parts[1]),
        default => throw new \InvalidArgumentException("Unknown content resource ID: " . implode('/', $parts)),
      },
      default => throw new \InvalidArgumentException("Unknown event platform resource: $parts[0]"),
    };
  }

  /**
   * Read content item.
   */
  private function readContentItem(array $contentType, string $itemId): array {
    $node = $this->entityTypeManager->getStorage('node')->getQuery()
      ->accessCheck()
      ->condition('nid', $itemId)
      ->condition('type', $contentType, 'IN')
      ->condition('status', 1)
      ->execute();

    if (empty($node)) {
      throw new \InvalidArgumentException("Content item not found: $itemId");
    }

    /** @var \Drupal\node\NodeInterface $nodeEntity */
    $nodeEntity = $this->entityTypeManager->getStorage('node')
      ->load(reset($node));
    $sessionData = $this->eventPlatformService->formatSessionData($nodeEntity, TRUE);

    return [
      new Resource(
        uri: "session/$itemId",
        name: $nodeEntity->getTitle(),
        description: NULL,
        mimeType: 'application/json',
        text: json_encode($sessionData, JSON_UNESCAPED_UNICODE),
      ),
    ];
  }

}
