<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Repository;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Repository for configuration-related database operations.
 */
final class ConfigRepository implements ConfigRepositoryInterface {

  /**
   * The entity type manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * @var \Psr\Log\LoggerInterface
   */
  private LoggerInterface $logger;

  /**
   * Constructs a new ConfigRepository.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   The logger channel factory.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, LoggerChannelFactoryInterface $loggerChannelFactory) {
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $loggerChannelFactory->get('event_platform_mcp');
  }

  /**
   * {@inheritdoc}
   */
  public function getEventDetails(): ?array {
    try {
      /** @var \Drupal\config_pages\ConfigPagesInterface|null $configPage */
      $configPage = $this->entityTypeManager->getStorage('config_pages')
        ->load('event_details');

      if (!$configPage) {
        return NULL;
      }

      return [
        'name' => $configPage->get('field_event_name')->value ?? NULL,
        'year' => $configPage->get('field_event_year')->value ?? date('Y'),
        'location' => $configPage->get('field_event_location_short')->value ?? NULL,
        'description' => $configPage->get('field_hompage_description_text')->value ?? '',
        'schedule_published' => $configPage->get('field_schedule_published')->value ?? FALSE,
        'submission_deadline' => $configPage->get('field_session_submission_deadlin')->value ?? NULL,
        'dates' => $this->getEventDateRange(),
      ];
    }
    catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      $this->logger->error('Error retrieving event details: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEventDateRange(): ?array {
    try {
      /** @var \Drupal\config_pages\ConfigPagesInterface|null $configPage */
      $configPage = $this->entityTypeManager->getStorage('config_pages')
        ->load('event_details');

      if (!$configPage || !$configPage->hasField('field_dates') || $configPage->get('field_dates')->isEmpty()) {
        return NULL;
      }

      $dateField = $configPage->get('field_dates')->first()->getValue();
      return [
        'start' => $dateField['value'],
        'end' => $dateField['end_value'],
      ];
    }
    catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      $this->logger->error('Error retrieving event date range: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function findUserIdByName(string $name): ?string {
    try {
      $users = $this->entityTypeManager->getStorage('user')
        ->loadByProperties(['name' => $name]);

      if (!empty($users)) {
        return (string) reset($users)->id();
      }
    }
    catch (PluginNotFoundException | InvalidPluginDefinitionException $e) {
      $this->logger->error('Error finding user by name: @message', ['@message' => $e->getMessage()]);
    }

    return NULL;
  }

}
