<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Service;

/**
 * Service for date/time operations and validation.
 */
final class DateService implements DateServiceInterface {

  /**
   * {@inheritdoc}
   */
  public function normalizeDateTime(string $dateTime): string {
    // If it's just a date (YYYY-MM-DD), add default time.
    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateTime)) {
      return $dateTime . ' 00:00:00';
    }

    // If it's date with time (YYYY-MM-DD HH:MM), add seconds.
    if (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/', $dateTime)) {
      return $dateTime . ':00';
    }

    return $dateTime;
  }

  /**
   * {@inheritdoc}
   */
  public function validateDateTimeInEventRange(string $dateTime, ?array $eventRange): bool {
    if (!$eventRange) {
      // If no event range is configured, allow any date.
      return TRUE;
    }

    $minDateTime = $eventRange['min'] . ' 00:00:00';
    $maxDateTime = $eventRange['max'] . ' 23:59:59';

    return $dateTime >= $minDateTime && $dateTime <= $maxDateTime;
  }

  /**
   * {@inheritdoc}
   */
  public function validateDateRange(array $dateRangeArgs, ?array $eventRange): array {
    $errors = [];

    // Check if at least one parameter is provided.
    if (empty($dateRangeArgs['start']) && empty($dateRangeArgs['end'])) {
      $errors[] = 'At least one of start or end must be provided in date_range';
      return ['valid' => FALSE, 'errors' => $errors];
    }

    // Validate start date if provided.
    if (!empty($dateRangeArgs['start'])) {
      $startDate = $this->normalizeDateTime($dateRangeArgs['start']);
      if (!$this->validateDateTimeInEventRange($startDate, $eventRange)) {
        $errors[] = "Start date '{$dateRangeArgs['start']}' is outside the event date range";
      }
    }

    // Validate end date if provided.
    if (!empty($dateRangeArgs['end'])) {
      $endDate = $this->normalizeDateTime($dateRangeArgs['end']);
      if (!$this->validateDateTimeInEventRange($endDate, $eventRange)) {
        $errors[] = "End date '{$dateRangeArgs['end']}' is outside the event date range";
      }
    }

    // Validate that start is before end if both are provided.
    if (!empty($dateRangeArgs['start']) && !empty($dateRangeArgs['end'])) {
      $startDate = $this->normalizeDateTime($dateRangeArgs['start']);
      $endDate = $this->normalizeDateTime($dateRangeArgs['end']);
      if ($startDate > $endDate) {
        $errors[] = 'Start date must be before or equal to end date';
      }
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function formatTimeSlotData(array $whenFieldValue): array {
    return [
      'start' => date('Y-m-d H:i:s', (int) $whenFieldValue['value']),
      'end' => $whenFieldValue['end_value'] ? date('Y-m-d H:i:s', (int) $whenFieldValue['end_value']) : NULL,
      'duration' => $whenFieldValue['duration'] ?? 60,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function filterByDate(array $timeSlotData, ?string $date): bool {
    if (!$date || !$timeSlotData['when']) {
      return TRUE;
    }

    $slotDate = date('Y-m-d', strtotime($timeSlotData['when']['start']));
    return $slotDate === $date;
  }

  /**
   * {@inheritdoc}
   */
  public function sortByStartTime(array $schedule): array {
    usort($schedule, static function ($a, $b) {
      if (!$a['when'] || !$b['when']) {
        return 0;
      }
      return strtotime($a['when']['start']) <=> strtotime($b['when']['start']);
    });

    return $schedule;
  }

}
