<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Service;

/**
 * Interface for date/time operations and validation.
 */
interface DateServiceInterface {

  /**
   * Normalize date/time string to include time if not present.
   *
   * @param string $dateTime
   *   The date/time string to normalize.
   *
   * @return string
   *   Normalized date/time string.
   */
  public function normalizeDateTime(string $dateTime): string;

  /**
   * Validate date/time is within event range.
   *
   * @param string $dateTime
   *   The date/time to validate.
   * @param array|null $eventRange
   *   The event date range.
   *
   * @return bool
   *   TRUE if within range, FALSE otherwise.
   */
  public function validateDateTimeInEventRange(string $dateTime, ?array $eventRange): bool;

  /**
   * Validate date range parameters.
   *
   * @param array $dateRangeArgs
   *   The date range arguments to validate.
   * @param array|null $eventRange
   *   The event date range.
   *
   * @return array
   *   Array with 'valid' boolean and 'errors' array.
   */
  public function validateDateRange(array $dateRangeArgs, ?array $eventRange): array;

  /**
   * Format time slot data from field value.
   *
   * @param array $whenFieldValue
   *   The field value from smart date field.
   *
   * @return array
   *   Formatted time slot data.
   */
  public function formatTimeSlotData(array $whenFieldValue): array;

  /**
   * Filter time slot by date.
   *
   * @param array $timeSlotData
   *   The time slot data to filter.
   * @param string|null $date
   *   The date to filter by (YYYY-MM-DD format).
   *
   * @return bool
   *   TRUE if time slot matches the date, FALSE otherwise.
   */
  public function filterByDate(array $timeSlotData, ?string $date): bool;

  /**
   * Sort schedule by start time.
   *
   * @param array $schedule
   *   The schedule array to sort.
   *
   * @return array
   *   Sorted schedule array.
   */
  public function sortByStartTime(array $schedule): array;

}
