<?php

declare(strict_types=1);

namespace Drupal\event_platform_mcp\Repository;

use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Repository for taxonomy-related database operations.
 */
final class TaxonomyRepository implements TaxonomyRepositoryInterface {

  /**
   * The entity type manager.
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new TaxonomyRepository.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function getTermsByVocabulary(string $vocabulary): array {
    try {
      /** @var \Drupal\taxonomy\TermInterface[] $terms */
      return $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => $vocabulary]);
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function findTermIdByName(string $vocabulary, string $name): ?string {
    try {
      $terms = $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => $vocabulary, 'name' => $name]);

      if (!empty($terms)) {
        return (string) reset($terms)->id();
      }

      return NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTimeSlotsByDate(string $date): array {
    try {
      /** @var \Drupal\taxonomy\TermInterface[] $timeSlots */
      $timeSlots = $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => 'time_slot']);

      $relevantTimeSlots = [];
      foreach ($timeSlots as $timeSlot) {
        if ($timeSlot->hasField('field_when') && !$timeSlot->get('field_when')->isEmpty()) {
          /** @var \Drupal\smart_date\Plugin\Field\FieldType\SmartDateItem $whenField */
          $whenField = $timeSlot->get('field_when')->first();
          if ($whenField->start_date) {
            $slotDate = date('Y-m-d', strtotime($whenField->start_date->format('Y-m-d H:i:s')));
            if ($slotDate === $date) {
              $relevantTimeSlots[] = $timeSlot->id();
            }
          }
        }
      }

      return $relevantTimeSlots;
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTimeSlotsByDateRange(string $startDate, string $endDate): array {
    try {
      /** @var \Drupal\taxonomy\TermInterface[] $timeSlots */
      $timeSlots = $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => 'time_slot']);

      $relevantTimeSlots = [];
      foreach ($timeSlots as $timeSlot) {
        if ($timeSlot->hasField('field_when') && !$timeSlot->get('field_when')->isEmpty()) {
          /** @var \Drupal\smart_date\Plugin\Field\FieldType\SmartDateItem $whenField */
          $whenField = $timeSlot->get('field_when')->first();
          if (!$whenField->isEmpty()) {
            $when_field_value = $whenField->getValue();
            $slotDate = date('Y-m-d', strtotime($when_field_value['value']->format('Y-m-d H:i:s')));
            if ($slotDate >= $startDate && $slotDate <= $endDate) {
              $relevantTimeSlots[] = $timeSlot->id();
            }
          }
        }
      }

      return $relevantTimeSlots;
    }
    catch (\Exception $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getAllTimeSlots(): array {
    try {
      /** @var \Drupal\taxonomy\TermInterface[] $timeSlots */
      return $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => 'time_slot']);
    }
    catch (\Exception $e) {
      return [];
    }
  }

}
