<?php

namespace Drupal\event_to_calendar\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for Event to Calendar functionality.
 */
class EventToCalendarController extends ControllerBase {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Generate an iCal (.ics) file for an event.
   */
  public function generateIcal($event_id) {
    // Load the event node.
    $node = $this->entityTypeManager->getStorage('node')->load($event_id);

    // Ensure the node exists.
    if (!$node) {
      throw new NotFoundHttpException();
    }

    // Get configuration for field mappings.
    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_types = $config->get('content_types');

    // Ensure the current node's content type is allowed.
    $content_type = $node->getType();
    if (!in_array($content_type, array_keys($content_types))) {
      throw new NotFoundHttpException();
    }

    // Get the relevant field mappings from configuration.
    $start_date_field = $config->get($content_type . '_start_date');
    // Fallback to start date if end date is not set.
    $end_date_field = $config->get($content_type . '_end_date') ?: $start_date_field;
    $location_field = $config->get($content_type . '_location');

    // Generate iCal content based on the mapped fields.
    $ics_content = $this->generateIcsContent($node, $start_date_field, $end_date_field, $location_field);

    // Set headers for downloading the file.
    $response = new Response($ics_content);
    $response->headers->set('Content-Type', 'text/calendar');
    $response->headers->set('Content-Disposition', 'attachment; filename="event.ics"');

    return $response;
  }

  /**
   * Generate iCal content for the event.
   */
  private function generateIcsContent($node, $start_date_field, $end_date_field, $location_field) {
    // Retrieve date values from the node using dynamic field names.
    $start_date = $node->get($start_date_field)->value;
    $end_date = $node->get($end_date_field)->value;
    $location_field = $node->get($location_field)->value;
    // Format the dates for iCal.
    $formatted_start_date = $this->formatDateForIcal($start_date);
    $formatted_end_date = $this->formatDateForIcal($end_date);

    // Create the iCal content.
    $ics_content = "BEGIN:VCALENDAR\r\n";
    $ics_content .= "VERSION:2.0\r\n";
    $ics_content .= "BEGIN:VEVENT\r\n";
    $ics_content .= "SUMMARY:" . $node->getTitle() . "\r\n";
    $ics_content .= "DTSTART:" . $formatted_start_date . "\r\n";
    $ics_content .= "DTEND:" . $formatted_end_date . "\r\n";
    $ics_content .= "LOCATION:" . $location_field . "\r\n";
    $ics_content .= "END:VEVENT\r\n";
    $ics_content .= "END:VCALENDAR\r\n";

    return $ics_content;
  }

  /**
   * Format date for iCal.
   */
  private function formatDateForIcal($timestamp) {
    // Create a DateTime object from the timestamp in MST.
    // Use the timestamp.
    $dateTime = new \DateTime("@$timestamp");
    // Set to MST timezone.
    $dateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Convert to UTC.
    $dateTime->setTimezone(new \DateTimeZone('UTC'));

    // Format the date to Ymd\THis\Z (UTC format)
    return $dateTime->format('Ymd\THis\Z');
  }

  /**
   * Generate the RSS feed for upcoming events.
   */
  public function generateRssFeed($event_id) {
    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_types = $config->get('content_types');

    $rss = "<?xml version='1.0' encoding='UTF-8' ?>\r\n";
    $rss .= "<rss version='2.0'>\r\n";
    $rss .= "<channel>\r\n";
    $rss .= "<title>Upcoming Events</title>\r\n";
    $rss .= "<link>" . \Drupal::request()->getSchemeAndHttpHost() . "</link>\r\n";
    $rss .= "<description>List of upcoming events</description>\r\n";

    foreach ($content_types as $content_type => $enabled) {
      if ($enabled) {
        // Get the dynamic start date field from the configuration.
        $start_date_field = $config->get($content_type . '_start_date') ?: NULL;
        $end_date_field = $config->get($content_type . '_end_date') ?: NULL;
        $location_field = $config->get($content_type . '_location') ?: NULL;

        $query = $this->entityTypeManager->getStorage('node')->getQuery();
        $query->condition('status', 1)
          ->condition('type', $content_type)
          ->condition('nid', $event_id)
          ->sort($start_date_field, 'ASC')
          ->accessCheck(FALSE);

        // Execute the query and load nodes.
        $nids = $query->execute();
        $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($nids);

        // Loop through nodes and add them to the RSS feed.
        foreach ($nodes as $node) {
          $title = htmlspecialchars($node->getTitle());
          $link = \Drupal::request()->getSchemeAndHttpHost() . '/node/' . $node->id();
          $description = htmlspecialchars(strip_tags($node->get('body')->value));

          // Get the start date value and convert it to UTC.
          $start_date_value = $node->get($start_date_field)->value;
          $end_date_value = $node->get($end_date_field)->value;
          // Create a DateTime object from the timestamp in MST.
          // Use the timestamp.
          $startDateTime = new \DateTime("@$start_date_value");
          // Use the timestamp.
          $endDateTime = new \DateTime("@$end_date_value");
          // Set to MST timezone.
          $startDateTime->setTimezone(new \DateTimeZone('America/Denver'));
          // Set to MST timezone.
          $endDateTime->setTimezone(new \DateTimeZone('America/Denver'));

          // Convert to UTC.
          $startDateTime->setTimezone(new \DateTimeZone('UTC'));
          $endDateTime->setTimezone(new \DateTimeZone('UTC'));

          // Location.
          $location = $node->get($location_field)->value;

          // Format the publication date.
          // RFC 2822 format.
          $pub_date = $startDateTime->format('r');
          // RFC 2822 format.
          $end_date = $endDateTime->format('r');

          $rss .= "<item>\r\n";
          $rss .= "<title>{$title}</title>\r\n";
          $rss .= "<link>{$link}</link>\r\n";
          $rss .= "<description>{$description}</description>\r\n";
          $rss .= "<pubDate>{$pub_date}</pubDate>\r\n";
          $rss .= "<startDate>{$pub_date}</startDate>\r\n";
          $rss .= "<endDate>{$end_date}</endDate>\r\n";
          $rss .= "<Location>{$location}</Location>\r\n";
          $rss .= "</item>\r\n";
        }
      }
    }

    $rss .= "</channel>\r\n";
    $rss .= "</rss>\r\n";

    // Set the response headers and return the RSS feed.
    $response = new Response($rss);
    $response->headers->set('Content-Type', 'application/rss+xml');
    return $response;
  }

  /**
   * Generate the link for Google Calendar.
   */
  public function getGoogleCalendarUrl($event_id) {
    $node = $this->entityTypeManager->getStorage('node')->load($event_id);
    if (!$node) {
      throw new NotFoundHttpException();
    }

    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_type = $node->getType();
    $start_date_field = $config->get($content_type . '_start_date') ?: NULL;
    $end_date_field = $config->get($content_type . '_end_date') ?: NULL;
    // Assuming this is a timestamp in MST.
    $start_date = $node->get($start_date_field)->value;
    // Assuming this is a timestamp in MST.
    $end_date = $node->get($end_date_field)->value;
    $location_field = $config->get($content_type . '_location') ?: NULL;
    $body = $node->get('body')->value;
    // Create a DateTime object from the timestamp in MST.
    // Use the timestamp.
    $startDateTime = new \DateTime("@$start_date");
    // Set to MST timezone.
    $startDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Use the timestamp.
    $endDateTime = new \DateTime("@$end_date");
    // Set to MST timezone.
    $endDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Convert to UTC.
    $startDateTime->setTimezone(new \DateTimeZone('UTC'));
    $endDateTime->setTimezone(new \DateTimeZone('UTC'));

    // Format the date to Ymd\THis\Z.
    $formatted_start_date = $startDateTime->format('Ymd\THis\Z');
    $formatted_end_date = $endDateTime->format('Ymd\THis\Z');

    // Location.
    $location = $node->get($location_field)->value;

    // Generate the link.
    $url = "https://www.google.com/calendar/render?action=TEMPLATE&text=" . urlencode($node->getTitle()) . "&dates={$formatted_start_date}/{$formatted_end_date}" . "&location=" . urlencode($location) . "&details=" . urlencode($body);
    return new TrustedRedirectResponse($url);
  }

  /**
   * Generate the link for Outlook Calendar.
   */
  public function getOutlookCalendarUrl($event_id) {
    $node = $this->entityTypeManager->getStorage('node')->load($event_id);
    if (!$node) {
      throw new NotFoundHttpException();
    }

    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_type = $node->getType();
    $start_date_field = $config->get($content_type . '_start_date');
    $end_date_field = $config->get($content_type . '_end_date') ?: $start_date_field;
    $location_field = $config->get($content_type . '_location') ?: NULL;

    $start_date = $node->get($start_date_field)->value;
    $end_date = $node->get($end_date_field)->value;

    // Create a DateTime object from the timestamp in MST.
    // Use the timestamp.
    $startDateTime = new \DateTime("@$start_date");
    // Set to MST timezone.
    $startDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Use the timestamp.
    $endDateTime = new \DateTime("@$end_date");
    // Set to MST timezone.
    $endDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Convert to UTC.
    $startDateTime->setTimezone(new \DateTimeZone('UTC'));
    $endDateTime->setTimezone(new \DateTimeZone('UTC'));

    // Format the date to Ymd\THis\Z.
    $formatted_start_date = $startDateTime->format('Ymd\THis\Z');
    $formatted_end_date = $endDateTime->format('Ymd\THis\Z');

    // Location.
    $location = $node->get($location_field)->value;

    $url = "https://outlook.live.com/calendar/0/deeplink/compose?subject=" . urlencode($node->getTitle()) . "&startdt={$formatted_start_date}&enddt={$formatted_end_date}&allday=false" . "&location=" . urlencode($location);
    return new TrustedRedirectResponse($url);
  }

  /**
   * Generate the link for Yahoo Calendar.
   */
  public function getYahooCalendarUrl($event_id) {
    $node = $this->entityTypeManager->getStorage('node')->load($event_id);
    if (!$node) {
      throw new NotFoundHttpException();
    }

    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_type = $node->getType();
    $start_date_field = $config->get($content_type . '_start_date');
    $end_date_field = $config->get($content_type . '_end_date') ?: $start_date_field;
    $location_field = $config->get($content_type . '_location') ?: NULL;

    $start_date = $node->get($start_date_field)->value;
    $end_date = $node->get($end_date_field)->value;

    // Create a DateTime object from the timestamp in MST.
    // Use the timestamp.
    $startDateTime = new \DateTime("@$start_date");
    // Set to MST timezone.
    $startDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Use the timestamp.
    $endDateTime = new \DateTime("@$end_date");
    // Set to MST timezone.
    $endDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Convert to UTC.
    $startDateTime->setTimezone(new \DateTimeZone('UTC'));
    $endDateTime->setTimezone(new \DateTimeZone('UTC'));

    // Format the date to Ymd\THis\Z.
    $formatted_start_date = $startDateTime->format('Ymd\THis\Z');
    $formatted_end_date = $endDateTime->format('Ymd\THis\Z');

    // Location.
    $location = $node->get($location_field)->value;

    $url = "https://calendar.yahoo.com/?v=60&view=d&type=20&title=" . urlencode($node->getTitle()) . "&st={$formatted_start_date}&et={$formatted_end_date}" . "&in_loc=" . urlencode($location);
    return new TrustedRedirectResponse($url);
  }

  /**
   * Generate the link for vcs for android.
   */
  public function generateVcs($event_id) {
    // Load the event node.
    $node = $this->entityTypeManager->getStorage('node')->load($event_id);

    // Ensure the node exists.
    if (!$node) {
      throw new NotFoundHttpException();
    }

    // Get configuration for field mappings.
    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_type = $node->getType();
    $start_date_field = $config->get($content_type . '_start_date');
    $end_date_field = $config->get($content_type . '_end_date') ?: $start_date_field;
    $location_field = $config->get($content_type . '_location') ?: NULL;

    // Get start and end dates.
    $start_date = $node->get($start_date_field)->value;
    $end_date = $node->get($end_date_field)->value;

    // Format dates for vCalendar.
    // Create a DateTime object from the timestamp in MST.
    // Use the timestamp.
    $startDateTime = new \DateTime("@$start_date");
    // Set to MST timezone.
    $startDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Use the timestamp.
    $endDateTime = new \DateTime("@$end_date");
    // Set to MST timezone.
    $endDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Convert to UTC.
    $startDateTime->setTimezone(new \DateTimeZone('UTC'));
    $endDateTime->setTimezone(new \DateTimeZone('UTC'));

    // Format the date to Ymd\THis\Z.
    $formatted_start_date = $startDateTime->format('Ymd\THis\Z');
    $formatted_end_date = $endDateTime->format('Ymd\THis\Z');

    // Location.
    $location = $node->get($location_field)->value;

    // Generate vCalendar content.
    $vcs_content = "BEGIN:VCALENDAR\r\n";
    $vcs_content .= "VERSION:1.0\r\n";
    $vcs_content .= "BEGIN:VEVENT\r\n";
    $vcs_content .= "SUMMARY:" . $node->getTitle() . "\r\n";
    $vcs_content .= "DTSTART:$formatted_start_date\r\n";
    $vcs_content .= "DTEND:$formatted_end_date\r\n";
    $vcs_content .= "DESCRIPTION:" . strip_tags($node->get('body')->value) . "\r\n";
    $vcs_content .= "LOCATION:" . $location . "\r\n";
    $vcs_content .= "END:VEVENT\r\n";
    $vcs_content .= "END:VCALENDAR\r\n";

    // Set headers for downloading the file.
    $response = new Response($vcs_content);
    $response->headers->set('Content-Type', 'text/x-vcalendar');
    $response->headers->set('Content-Disposition', 'attachment; filename="event.vcs"');

    return $response;
  }

  /**
   * Generate the csv content.
   */
  public function generateCsv($event_id) {
    // Load the event node.
    $node = $this->entityTypeManager->getStorage('node')->load($event_id);

    // Ensure the node exists.
    if (!$node) {
      throw new NotFoundHttpException();
    }

    // Get configuration for field mappings.
    $config = $this->configFactory->get('event_to_calendar.settings');
    $content_type = $node->getType();
    $start_date_field = $config->get($content_type . '_start_date');
    $end_date_field = $config->get($content_type . '_end_date') ?: NULL;
    $location_field = $config->get($content_type . '_location') ?: NULL;

    // Get start and end dates.
    $start_date = date('Y-m-d H:i:s', strtotime($node->get($start_date_field)->value));
    $end_date = date('Y-m-d H:i:s', strtotime($node->get($end_date_field)->value));
    // Create a DateTime object from the timestamp in MST.
    // Use the timestamp.
    $startDateTime = new \DateTime("@$start_date");
    // Set to MST timezone.
    $startDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Use the timestamp.
    $endDateTime = new \DateTime("@$end_date");
    // Set to MST timezone.
    $endDateTime->setTimezone(new \DateTimeZone('America/Denver'));

    // Convert to UTC.
    $startDateTime->setTimezone(new \DateTimeZone('UTC'));
    $endDateTime->setTimezone(new \DateTimeZone('UTC'));

    // Format the date to Ymd\THis\Z.
    $formatted_start_date = $startDateTime->format('Ymd\THis\Z');
    $formatted_end_date = $endDateTime->format('Ymd\THis\Z');

    // Location.
    $location = $node->get($location_field)->value;

    // Prepare CSV content.
    $csv_content = [
      ['Event Title', 'Start Date', 'End Date', 'Description', 'Location'],
      [$node->getTitle(), $formatted_start_date, $formatted_end_date, strip_tags($node->get('body')->value) , $location],
    ];

    // Convert the CSV array to a string.
    $output = fopen('php://temp', 'r+');
    foreach ($csv_content as $row) {
      fputcsv($output, $row);
    }
    rewind($output);
    $csv_string = stream_get_contents($output);
    fclose($output);

    // Set headers for downloading the file.
    $response = new Response($csv_string);
    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', 'attachment; filename="event.csv"');

    return $response;
  }

}
