<?php

namespace Drupal\event_to_calendar\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form.
 */
class EventToCalendarConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'event_to_calendar_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['event_to_calendar.settings'];
  }

  /**
   * Build the configuration form.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('event_to_calendar.settings');

    // Fetch available content types.
    $content_types = \Drupal::service('entity_type.manager')->getStorage('node_type')->loadMultiple();
    $options = [];
    foreach ($content_types as $type) {
      $options[$type->id()] = $type->label();
    }

    // Content types checkboxes.
    $form['content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content Types for Calendar Integration'),
      '#options' => $options,
      '#default_value' => $config->get('content_types') ?? [],
      '#description' => $this->t('Select the content types you want to enable for calendar integration.'),
    ];

    // Field mappings.
    $form['field_mappings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Field Mappings'),
    ];

    foreach ($options as $content_type => $label) {
      if (!empty($content_type)) {
        $form['field_mappings'][$content_type . '_start_date'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Start Date Field for @type', ['@type' => $label]),
          '#default_value' => $config->get($content_type . '_start_date'),
          '#description' => $this->t('Enter the field machine name for the start date.'),
          '#states' => [
            'visible' => [
              ':input[name="content_types[' . $content_type . ']"]' => ['checked' => TRUE],
            ],
          ],
        ];

        $form['field_mappings'][$content_type . '_end_date'] = [
          '#type' => 'textfield',
          '#title' => $this->t('End Date Field for @type', ['@type' => $label]),
          '#default_value' => $config->get($content_type . '_end_date'),
          '#description' => $this->t('Enter the field machine name for the end date (optional).'),
          '#states' => [
            'visible' => [
              ':input[name="content_types[' . $content_type . ']"]' => ['checked' => TRUE],
            ],
          ],
        ];

        $form['field_mappings'][$content_type . '_location'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Location Field for @type', ['@type' => $label]),
          '#default_value' => $config->get($content_type . '_location'),
          '#description' => $this->t('Enter the field machine name for the location.'),
          '#states' => [
            'visible' => [
              ':input[name="content_types[' . $content_type . ']"]' => ['checked' => TRUE],
            ],
          ],
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);

    // Save the selected content types and their respective field settings.
    $this->config('event_to_calendar.settings')
      ->set('content_types', array_filter($form_state->getValue('content_types')))
      ->save();

    foreach ($form_state->getValues() as $key => $value) {
      if (strpos($key, '_start_date') !== FALSE || strpos($key, '_end_date') !== FALSE || strpos($key, '_location') !== FALSE) {
        $this->config('event_to_calendar.settings')
          ->set($key, $value)
          ->save();
      }
    }
  }

}
