<?php

namespace Drupal\event_to_calendar\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an 'Add to Calendar' Block.
 *
 * @Block(
 *   id = "event_to_calendar_block",
 *   admin_label = @Translation("Add to Calendar Block"),
 *   category = @Translation("Custom")
 * )
 */
class AddToCalendarBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AddToCalendarBlock.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Get the current node.
    $node = \Drupal::routeMatch()->getParameter('node');

    // Check if the node is available.
    if ($node) {
      $event_id = $node->id();
      $content_type = $node->getType();

      // Load the configuration settings.
      $config = $this->configFactory->get('event_to_calendar.settings');

      // Check if the current content type is enabled in the configuration.
      $content_types = $config->get('content_types') ?: [];
      if (isset($content_types[$content_type]) && $content_types[$content_type]) {

        // Fetch field for start and end date, location from configuration.
        $start_date_field = $config->get($content_type . '_start_date') ?: NULL;
        $end_date_field = $config->get($content_type . '_end_date') ?: $start_date_field;
        $location_field = $config->get($content_type . '_location') ?: NULL;

        // Ensure the necessary fields are available.
        if ($start_date_field && $location_field) {
          // Fetch the relevant field values.
          $start_date = $node->get($start_date_field)->value;
          $end_date = $node->get($end_date_field)->value;
          $location = $node->get($location_field)->value;

          // Build URLs for each calendar platform.
          $ical_url = '/event/' . $event_id . '/ical';
          $google_calendar_url = '/event/' . $event_id . '/google';
          $outlook_url = '/event/' . $event_id . '/outlook';
          $yahoo_calendar_url = '/event/' . $event_id . '/yahoo';
          $rss_url = '/event/' . $event_id . '/rss';
          $vcs_url = '/event/' . $event_id . '/vcs';
          $csv_url = '/event/' . $event_id . '/csv';

          // Prepare data for the Twig template.
          $data = [
            'button_label' => $this->t('Add to Calendar'),
            'modal_title' => "Subscribe to the Schedule",
            'modal_description' => $node->get('body')->value ?? 'Event description not available.',
            'ical_url' => $ical_url,
            'google_calendar_url' => $google_calendar_url,
            'outlook_url' => $outlook_url,
            'yahoo_calendar_url' => $yahoo_calendar_url,
            'vcs_url' => $vcs_url,
            'csv_url' => $csv_url,
            'rss_url' => $rss_url,
          ];

          // Render the template.
          return [
            '#theme' => 'event_to_calendar',
            '#data' => $data,
            '#attached' => [
              'library' => [
                'event_to_calendar/modal',
              ],
            ],
          ];
        }
      }
    }
    return [];
  }

}
