<?php

namespace Drupal\exact_online\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\exact_online\Service\ExactOnlineServiceInterface;

/**
 * Controller for Exact Online authorization.
 */
class ExactOnlineAuthController extends ControllerBase {

  /**
   * The http request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The Exact Online service.
   *
   * @var \Drupal\exact_online\Service\ExactOnlineServiceInterface
   */
  protected ExactOnlineServiceInterface $exactOnlineService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->request = $container->get('request_stack')->getCurrentRequest();
    $instance->exactOnlineService = $container->get('exact_online.service');
  }

  /**
   * Constructs an ExactOnlineAuthController object.
   *
   * @param \Drupal\exact_online\Service\ExactOnlineServiceInterface $exact_online_service
   *   The Exact Online service.
   */
  public function __construct(ExactOnlineServiceInterface $exact_online_service) {
    $this->exactOnlineService = $exact_online_service;
  }

  /**
   * Initiates the OAuth authorization flow.
   *
   * @return \Drupal\Core\Routing\TrustedRedirectResponse|\Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect.
   */
  public function authorize(): TrustedRedirectResponse|RedirectResponse {
    try {
      $auth_url = $this->exactOnlineService->getAuthorizationUrl();
      // Use TrustedRedirectResponse for external URL.
      return new TrustedRedirectResponse($auth_url);
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Authorization failed: @error', [
        '@error' => $e->getMessage(),
      ]));
      return $this->redirect('exact_online.dashboard');
    }
  }

  /**
   * Handles the OAuth callback.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect.
   */
  public function callback(): RedirectResponse {
    try {
      $this->exactOnlineService->handleCallback();
      $this->messenger()->addMessage($this->t('Successfully connected to Exact Online at @date', [
        '@date' => date('d-m-Y H:i:s'),
      ]));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Connection failed: @error', [
        '@error' => $e->getMessage(),
      ]));
    }

    return $this->redirect('exact_online.dashboard');
  }

  /**
   * Reset connection.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|array
   *   Redirect or array.
   */
  public function reset(): RedirectResponse|array {
    try {
      $params = $this->request->query->all();
      if (isset($params['confirm']) && $params['confirm'] === '1') {
        if ($this->exactOnlineService->getConnection()) {
          $this->exactOnlineService->resetConnection();
          $this->messenger()->addWarning($this->t('Connection with Exact Online was reset (you need to re-authorize)'));
        }
      }
      $is_dialog = (bool) $this->request->get('js');
      if ($is_dialog && isset($params['dialogOptions'])) {
        // Return content for the confirmation modal.
        $warning_msg = '
          <h3>Weet je het zeker dat je de koppeling met Exact Online wilt resetten? Alle opgeslagen tokens zullen verwijderd worden.</h3>
        ';
        $confirm_link = Url::fromRoute('exact_online.reset');
        $markup = Markup::create(
          $warning_msg . '<a href="' . $confirm_link->toString() . '?confirm=1" class="button button--danger">Yes</a>'
        );
        $build = [
          '#type' => 'markup',
          '#markup' => $markup,
        ];
        return $build;
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Connection reset with Exact Online failed: @error', [
        '@error' => $e->getMessage(),
      ]));
    }

    return $this->redirect('exact_online.dashboard');
  }

}
