<?php

namespace Drupal\exact_online\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\exact_online\Service\ExactOnlineServiceInterface;

/**
 * Controller for the Exact Online dashboard.
 */
class ExactOnlineDashboardController extends ControllerBase {

  /**
   * The Exact Online service.
   *
   * @var \Drupal\exact_online\Service\ExactOnlineServiceInterface
   */
  protected ExactOnlineServiceInterface $exactOnlineService;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->exactOnlineService = $container->get('exact_online.service');
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * Displays the Exact Online dashboard.
   *
   * @return array
   *   A render array for the dashboard.
   */
  public function dashboard(): array {
    $build = [];

    /* Connection */
    $build['status'] = [
      '#type' => 'details',
      '#title' => $this->t('Connection'),
      '#open' => TRUE,
    ];

    $config = $this->config('exact_online.settings');
    $client_id = $config->get('client_id');
    if (is_null($client_id)) {
      $msg_string = $this->t('Client ID is not set.');
      $message = Markup::create('<div class="messages messages--error">' . $msg_string . '</div>');
      $build['status']['connection'][] = [
        '#type' => 'markup',
        '#markup' => $message,
      ];
    }
    $client_secret = $this->state->get('exact_online.client_secret');
    if (is_null($client_secret)) {
      $msg_string = $this->t('Client secret is not set.');
      $message = Markup::create('<div class="messages messages--error">' . $msg_string . '</div>');
      $build['status']['connection'][] = [
        '#type' => 'markup',
        '#markup' => $message,
      ];
    }
    if (is_null($client_id) || is_null($client_secret)) {
      $msg_string = $this->t(
        '<a href="@settings_url">Click here</a> to configure the Exact Online settings.',
        [
          '@settings_url' => Url::fromRoute('exact_online.settings')->toString(),
        ]
      );
      $message = Markup::create('<div class="messages messages--info">' . $msg_string . '</div>');
      $build['status']['connection'][] = [
        '#type' => 'markup',
        '#markup' => $message,
      ];
    }

    if ($client_id && $client_secret) {
      $connection_status = $this->exactOnlineService->getConnectionStatus();
    }

    if (isset($connection_status)) {
      $status = $connection_status ?
        $this->t('Connected with Exact Online') :
        $this->t(
          'Not connected to Exact Online, click here to <a href="@authorize_url">authorize</a>.',
          ['@authorize_url' => Url::fromRoute('exact_online.authorize')->toString()]
        );
      $class_status = $connection_status ? 'messages--status' : 'messages--error';
      $message = Markup::create('<div class="messages ' . $class_status . '">' . $status . '</div>');
      $build['status']['connection'] = [
        '#type' => 'markup',
        '#markup' => $message,
      ];
    }

    /* Settings */
    $build['actions'] = [
      '#type' => 'details',
      '#title' => $this->t('Settings'),
      '#open' => TRUE,
    ];
    // @todo show checklist
    // - Client ID is set
    // - Client Secret is set
    // - Connection status.
    $build['actions']['configure'] = [
      '#type' => 'link',
      '#title' => $this->t('Configure Settings'),
      '#url' => Url::fromRoute('exact_online.settings'),
      '#attributes' => ['class' => ['button']],
    ];
    $build['actions']['logs'] = [
      '#type' => 'link',
      '#title' => $this->t('View Logs'),
      '#url' => Url::fromRoute('exact_online.logs'),
      '#attributes' => ['class' => ['button']],
    ];
    if (isset($connection_status)) {
      if ($connection_status === FALSE) {
        // Authorize link.
        $build['actions']['authorize'] = [
          '#type' => 'link',
          '#title' => $this->t('Authorize with Exact Online'),
          '#url' => Url::fromRoute('exact_online.authorize'),
          '#attributes' => ['class' => ['button', 'button--primary']],
        ];
      }
      else {
        // Reset link.
        $build['actions']['reset'] = [
          '#type' => 'link',
          '#title' => $this->t('Reset / logout'),
          '#url' => Url::fromRoute('exact_online.reset'),
          '#attributes' => [
            'class' => ['button'],
          ],
          '#ajax' => [
            'dialogType' => 'modal',
            'dialog' => ['height' => 400, 'width' => 700],
          ],
        ];
      }
    }

    /* Logs */
    $build['recent_activity'] = [
      '#type' => 'details',
      '#title' => $this->t('Last 10 log entries'),
      '#open' => TRUE,
    ];

    $recent_logs = $this->exactOnlineService->getRecentLogs();
    if (!empty($recent_logs)) {
      $build['recent_activity']['logs'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('Date'),
          $this->t('Type'),
          $this->t('Message'),
        ],
        '#rows' => array_map(function ($log) {
          return [
            $log['timestamp'],
            $log['type'],
            $log['message'],
          ];
        }, $recent_logs),
      ];
    }
    else {
      $build['recent_activity']['empty'] = [
        '#markup' => $this->t('No recent activity found.'),
      ];
    }

    return $build;
  }

}
