<?php

namespace Drupal\exact_online\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for Exact Online logs.
 */
class ExactOnlineLogsController extends ControllerBase {

  /**
   * The http request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * Entity type manager object.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * The Exact Online service.
   *
   * @var \Drupal\exact_online\Service\ExactOnlineServiceInterface
   */
  protected $exactOnlineService;

  /**
   * Formbuilder object.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->request = $container->get('request_stack')->getCurrentRequest();
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->exactOnlineService = $container->get('exact_online.service');
    $instance->formBuilder = $container->get('form_builder');
    return $instance;

  }

  /**
   * Displays the logs overview page.
   *
   * @return array
   *   A render array for the logs page.
   */
  public function overview() {
    $build = [];

    // Get filters from URL parameters.
    $filters = $this->request->query->all();

    $build['filters'] = [
      '#type' => 'details',
      '#title' => $this->t('Filter Logs'),
      '#open' => !empty($filters),
    ];

    $build['filters']['form'] = $this->formBuilder
      ->getForm('Drupal\exact_online\Form\ExactOnlineLogsFilterForm');

    $header = [
      'timestamp' => [
        'data' => $this->t('Date'),
        'field' => 'timestamp',
        'sort' => 'desc',
      ],
      'type' => [
        'data' => $this->t('Type'),
        'field' => 'type',
      ],
      'message' => $this->t('Message'),
      'user' => [
        'data' => $this->t('User'),
        'field' => 'user',
      ],
    ];

    $logs = $this->exactOnlineService->getLogs($filters);

    $rows = [];

    foreach ($logs as $log) {
      $rows[] = [
        'timestamp' => [
          'data' => [
            '#markup' => $log['timestamp'],
          ],
        ],
        'type' => [
          'data' => [
            '#markup' => $log['type'],
          ],
        ],
        'message' => [
          'data' => [
            '#markup' => $log['message'],
          ],
        ],
        'user' => [
          'data' => [
            '#markup' => $this->getUserName($log['user']),
          ],
        ],
      ];
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No logs available.'),
      '#attributes' => [
        'class' => ['exact-online-logs-table'],
      ],
    ];

    // @todo add button + callback to clear all log entries
    return $build;
  }

  /**
   * Gets the username for a user ID.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return string
   *   The username or 'Anonymous' if not found.
   */
  protected function getUserName(int $uid): string {
    try {
      if ($user = $this->entityTypeManager->getStorage('user')->load($uid)) {
        return $user->getDisplayName();
      }
    }
    catch (\Exception $e) {
      throw new \RuntimeException('Unable to retrieve user information.');
    }
    return $this->t('Anonymous');
  }

}
