<?php

namespace Drupal\exact_online\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for filtering Exact Online logs.
 */
class ExactOnlineLogsFilterForm extends FormBase {

  /**
   * The http request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->request = $container->get('request_stack')->getCurrentRequest();
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'exact_online_logs_filter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['#method'] = 'get';

    // Get current filter values from URL parameters.
    $filters = $this->request->query->all();

    $form['filters'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['container-inline'],
      ],
    ];

    $form['filters']['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Type'),
      '#options' => [
        '' => $this->t('- Any -'),
        'info' => $this->t('Info'),
        'error' => $this->t('Error'),
        'warning' => $this->t('Warning'),
      ],
      '#default_value' => $filters['type'] ?? '',
    ];

    $form['filters']['date_range'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['container-inline'],
      ],
    ];

    $form['filters']['date_range']['start_date'] = [
      '#type' => 'date',
      '#title' => $this->t('Start Date'),
      '#default_value' => $filters['start_date'] ?? '',
    ];

    $form['filters']['date_range']['end_date'] = [
      '#type' => 'date',
      '#title' => $this->t('End Date'),
      '#default_value' => $filters['end_date'] ?? '',
    ];

    $form['filters']['actions'] = [
      '#type' => 'actions',
      '#attributes' => [
        'class' => ['container-inline'],
      ],
    ];

    $form['filters']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
      '#button_type' => 'primary',
    ];

    if (!empty($filters)) {
      $form['filters']['actions']['reset'] = [
        '#type' => 'link',
        '#title' => $this->t('Reset'),
        '#url' => Url::fromRoute('exact_online.logs'),
        '#attributes' => [
          'class' => ['button'],
        ],
      ];
    }

    // Don't validate GET forms.
    $form['#validate'] = [];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Validate date range if both dates are provided.
    $start_date = $form_state->getValue('start_date');
    $end_date = $form_state->getValue('end_date');

    if (!empty($start_date) && !empty($end_date)) {
      if (strtotime($start_date) > strtotime($end_date)) {
        $form_state->setErrorByName('start_date', $this->t('Start date must be before end date.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $filters = [];

    // Collect filter values.
    $type = $form_state->getValue('type');
    if (!empty($type)) {
      $filters['type'] = $type;
    }

    $start_date = $form_state->getValue('start_date');
    if (!empty($start_date)) {
      $filters['start_date'] = $start_date;
    }

    $end_date = $form_state->getValue('end_date');
    if (!empty($end_date)) {
      $filters['end_date'] = $end_date;
    }

    // Redirect to the current page with filter parameters.
    $form_state->setRedirect('exact_online.logs', [], [
      'query' => $filters,
    ]);
  }

}
