<?php

namespace Drupal\exact_online\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configures Exact Online settings for this site.
 */
class ExactOnlineSettingsForm extends ConfigFormBase {

  /**
   * The http request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a new ExactOnlineSettingsForm.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->request = $container->get('request_stack')->getCurrentRequest();
    $instance->state = $container->get('state');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'exact_online_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['exact_online.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('exact_online.settings');

    $form['api_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Settings'),
      '#collapsible' => FALSE,
    ];

    $form['api_settings']['intro'] = [
      '#type' => 'markup',
      '#markup' => $this->t('
        <p>
          Manage your Exact Online apps: <a href="https://apps.exactonline.com">Exact Online App Store</a>
        </p>
        '
      ),
    ];

    $form['api_settings']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $config->get('client_id'),
      '#required' => TRUE,
      '#description' => $this->t('The Client ID from your Exact Online application.'),
    ];

    $form['api_settings']['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client Secret'),
      '#default_value' => $this->state->get('exact_online.client_secret'),
      '#required' => TRUE,
      '#description' => $this->t('The Client Secret from your Exact Online application.'),
      '#attributes' => [
        'autocomplete' => 'off',
      ],
    ];

    $form['api_settings']['callback_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Authorization callback domain'),
      '#default_value' => $config->get('callback_url') ?: $this->getDefaultCallbackUrl(),
      '#required' => TRUE,
      '#description' => $this->t(
        'The callback domain for OAuth authentication (<a href="https://support.exactonline.com/community/s/knowledge-base#All-All-DNO-Content-oauth-eol-oauth-devstep2">documentation</a>).<br /><br />
               Make sure to configure this domain in your Exact Online app settings.
        <br />
        <ul>
          <li>Do not add an "/" after .com</li>
          <li>You must include https://</li>
          <li>The domain must be reachable for the Exact Online servers, otherwise request are not processed.</li>
          <li>You cannot use a localhost for callback URL</li>
        </ul>
        For live websites this is your live domain</br><br/>
        While developing you can use a tunneled domain that hits your local development machine.<br/>
        This can be done with https://localtunnel.github.io/www<br/>
        <code>lt --port 80 --local-host @host</code>
        <br /><br />
        Current callback URL is: <strong>@callback_url</strong><br />
        You must set the callback URL as a Redirect URI in your Exact Online app settings.',
        [
          '@host' => $this->request->getHost(),
          '@callback_url' => $config->get('callback_url') . Url::fromRoute('exact_online.callback')->toString(),
        ]
      ),
    ];

    $form['api_settings']['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Exact Online API base URL'),
      '#default_value' => $config->get('base_url') ?: 'https://start.exactonline.nl',
      '#required' => TRUE,
      '#description' => $this->t('The base URL for the Exact Online API. Default is for The Netherlands.'),
    ];

    $form['api_settings']['division'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Division'),
      '#default_value' => $config->get('division'),
      '#description' => $this->t('Your Exact Online division number. Leave empty to auto-detect.'),
      '#required' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate callback URL format.
    $callback_url = $form_state->getValue('callback_url');
    if (!filter_var($callback_url, FILTER_VALIDATE_URL)) {
      $form_state->setError($form['api_settings']['callback_url'], $this->t('The callback URL must be a valid URL.'));
    }
    if (preg_match('/(https):\/\/*/', $callback_url) === 0) {
      $form_state->setError($form['api_settings']['callback_url'], $this->t('The callback URL must start with https://.'));
    }
    if (preg_match('/^https:\/\/localhost/i', $callback_url)) {
      $form_state->setError($form['api_settings']['callback_url'], $this->t('You cannot use localhost as a callback URL.'));
    }

    // Validate base URL format.
    $base_url = $form_state->getValue('base_url');
    if (!filter_var($base_url, FILTER_VALIDATE_URL)) {
      $form_state->setError($form['api_settings']['base_url'], $this->t('The base URL must be a valid URL.'));
    }

    // Validate division format if provided.
    $division = $form_state->getValue('division');
    if (!empty($division) && !is_numeric($division)) {
      $form_state->setError($form['api_settings']['division'], $this->t('The division must be a number.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('exact_online.settings');

    $config
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('callback_url', $form_state->getValue('callback_url'))
      ->set('base_url', $form_state->getValue('base_url'))
      ->set('division', $form_state->getValue('division'))
      ->save();

    $this->state->set('exact_online.client_secret', $form_state->getValue('client_secret'));

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets the default callback URL based on the site URL.
   *
   * @return string
   *   The default callback URL.
   */
  protected function getDefaultCallbackUrl(): string {
    return $this->request->getSchemeAndHttpHost() . '/exact-online/callback';
  }

}
