<?php

namespace Drupal\exact_online\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Picqer\Financials\Exact\Connection;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for interacting with Exact Online.
 */
class ExactOnlineService implements ExactOnlineServiceInterface {

  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelFactoryInterface|LoggerChannelInterface $loggerFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The http request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * Current Drupal user.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $user;

  /**
   * The Exact Online connection.
   *
   * @var \Picqer\Financials\Exact\Connection|null
   */
  protected ?Connection $connection;

  /**
   * Constructs a new ExactOnlineService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request
   *   The request stack.
   * @param \Drupal\Core\Session\AccountProxyInterface $user
   *   Current Drupal user.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    MessengerInterface $messenger,
    CacheBackendInterface $cache,
    TimeInterface $time,
    TranslationInterface $string_translation,
    RequestStack $request,
    AccountProxyInterface $user,
  ) {
    $this->configFactory = $config_factory;
    $this->state = $state;
    $this->loggerFactory = $logger_factory->get('exact_online');
    $this->messenger = $messenger;
    $this->cache = $cache;
    $this->time = $time;
    $this->stringTranslation = $string_translation;
    $this->request = $request;
    $this->user = $user;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function getAuthorizationUrl(): string {
    try {
      return $this->getConnection()->getAuthUrl();
    }
    catch (\Exception $e) {
      $this->logError('Failed to generate authorization URL: ' . $e->getMessage());
      throw $e;
    }
  }

  /**
   * Callback function with authorization info.
   *
   * {@inheritdoc}
   *
   * @throws \Picqer\Financials\Exact\ApiException
   */
  public function handleCallback(): bool {
    try {
      $connection = $this->getConnection();

      if (!$this->request->get('code')) {
        $error = $this->request->get('error');
        throw new \RuntimeException('No authorization code received, error: ' . $error);
      }
      // Set authorization code with received code.
      $code = $this->request->get('code');
      $connection->setAuthorizationCode($code);

      // Connect.
      $connection->connect();
      $access_token = $connection->getAccessToken();

      // Active sleep upon hitting minutely rate limits.
      $connection->setWaitOnMinutelyRateLimitHit(TRUE);

      if (is_null($access_token)) {
        throw new \RuntimeException('Could not get access token with authorization code: ' . $code);
      }

      // Update and store the tokens.
      $this->updateAndStoreTokens();

      $this->logInfo('Successfully authenticated with Exact Online at ' . date('d-m-Y H:i:s'));
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logError('Authentication failed: ' . $e->getMessage());
      throw $e;
    }
  }

  /**
   * Gets the Exact Online connection instance.
   *
   * @return \Picqer\Financials\Exact\Connection
   *   The connection instance.
   */
  public function getConnection(): Connection {
    if (!isset($this->connection)) {
      $config = $this->configFactory->get('exact_online.settings');

      $this->connection = new Connection();
      $this->connection->setRedirectUrl($config->get('callback_url') . '/exact-online/callback');
      $this->connection->setExactClientId($config->get('client_id'));
      $this->connection->setExactClientSecret($this->state->get('exact_online.client_secret'));
      if ($division = $config->get('division')) {
        $this->connection->setDivision($division);
      }
      // Set sleep upon minutely rate limits.
      $this->connection->setWaitOnMinutelyRateLimitHit(TRUE);

      // Set the base URL if configured.
      if ($base_url = $config->get('base_url')) {
        $this->connection->setBaseUrl($base_url);
      }
      // Restore tokens if available.
      $this->restoreTokens();
    }
    return $this->connection;
  }

  /**
   * {@inheritdoc}
   */
  public function getConnectionStatus(): bool {
    try {
      $this->getConnection();
      if (!$this->connection->getRefreshToken()) {
        throw new \RuntimeException('Could not get refresh token, you need to authorize with Exact Online.');
      }
      // Token details.
      $tokenDetails = $this->getTokenExpirationDetails();
      if ($tokenDetails['status'] === 'expired') {
        $this->updateAndStoreTokens();
      }
      $reconnect_notification_timestamp = $this->state->get('exact_online.reconnect_notification');
      if (!is_null($reconnect_notification_timestamp)) {
        $this->state->delete('exact_online.reconnect_notification');
      }
      return TRUE;
    }
    catch (\Exception $e) {
      $this->deleteTokens();
      $this->clearApiRateLimits();
      $this->logError('Connection status error: ' . $e->getMessage());
      // Send email notification to customer to inform the connection
      // with Exact has to be reset manually.
      $reconnect_notification_timestamp = $this->state->get('exact_online.reconnect_notification');
      // If null or timestamp is set more than 24 hours ago
      // (so a reminder is sent daily to re-authorize)
      if (is_null($reconnect_notification_timestamp) || ((time() - $reconnect_notification_timestamp) > 86400)) {
        // @todo send notification with connection error to admin
      }
      return FALSE;
    }
  }

  /**
   * Reset the connection by deleting all tokens.
   *
   * @return bool
   *   Boolean.
   */
  public function resetConnection(): bool {
    $this->deleteTokens();
    $this->clearApiRateLimits();
    $this->logInfo('Connection has been reset / tokens are deleted.');
    return TRUE;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function getCompanyInfo(): array {
    try {
      if (!$this->getConnectionStatus()) {
        throw new \RuntimeException('Not connected to Exact Online');
      }
      return [];
    }
    catch (\Exception $e) {
      $this->logError('Failed to fetch company info: ' . $e->getMessage());
      throw $e;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getRecentLogs(int $limit = 10): array {
    return $this->getLogs(['limit' => $limit]);
  }

  /**
   * {@inheritdoc}
   */
  public function getLogs(array $filters = []): array {
    $logs = $this->state->get('exact_online.logs', []);

    // Apply filters.
    if (!empty($filters)) {
      if (isset($filters['type'])) {
        $logs = array_filter($logs, function ($log) use ($filters) {
          return $log['type'] === $filters['type'];
        });
      }

      if (isset($filters['start_date'])) {
        $logs = array_filter($logs, function ($log) use ($filters) {
          return strtotime($log['timestamp']) >= strtotime($filters['start_date']);
        });
      }

      if (isset($filters['end_date'])) {
        $logs = array_filter($logs, function ($log) use ($filters) {
          return strtotime($log['timestamp']) <= strtotime($filters['end_date']);
        });
      }
    }

    // Sort by timestamp descending.
    usort($logs, function ($a, $b) {
      return strtotime($b['timestamp']) - strtotime($a['timestamp']);
    });

    // Apply limit if specified.
    if (isset($filters['limit'])) {
      $logs = array_slice($logs, 0, $filters['limit']);
    }

    return $logs;
  }

  /**
   * Gets detailed token expiration information.
   *
   * @return array
   *   An array containing token expiration details.
   */
  public function getTokenExpirationDetails(): array {
    try {
      $tokenExpires = $this->connection->getTokenExpires();
      $currentTime = $this->time->getRequestTime();

      if (empty($tokenExpires)) {
        return [
          'status' => 'unknown',
          'message' => 'Token expiration time not available',
          'expires_at' => NULL,
          'remaining_hours' => NULL,
        ];
      }

      $remainingSeconds = $tokenExpires - $currentTime;
      $remainingMinutes = round($remainingSeconds / 60, 1);
      $remainingHours = round($remainingSeconds / 3600, 1);

      // Determine status.
      $status = 'valid';
      if ($remainingSeconds <= 0) {
        $status = 'expired';
      }
      // Please note, new access tokens can be requested after 9 minutes and
      // 30 seconds after you've received it.
      elseif ($remainingSeconds <= (9 * 60)) {
        $status = 'expiring_soon';
      }

      return [
        'status' => $status,
        'message' => $this->getExpirationMessage($status, $remainingMinutes),
        'expires_at' => date('Y-m-d H:i:s', $tokenExpires),
        'remaining_hours' => $remainingHours,
        'expires_timestamp' => $tokenExpires,
        'current_timestamp' => $currentTime,
      ];
    }
    catch (\Exception $e) {
      return [
        'status' => 'error',
        'message' => sprintf('Error getting token details: %s', $e->getMessage()),
        'expires_at' => NULL,
        'remaining_hours' => NULL,
      ];
    }
  }

  /**
   * Check API rate limits.
   *
   * @return void
   *   Void/
   */
  public function checkApiRateLimits(): void {
    if (is_null($this->state->get('exact_online.api_rate_limits.dailyLimitRemaining'))) {
      $this->setApiRateLimits();
    }
    if (is_null($this->state->get('exact_online.api_rate_limits.minutelyLimitRemaining'))) {
      $this->setApiRateLimits();
    }
    // Current time in milliseconds.
    //
    $time_in_ms = time() * 1000;
    if ($time_in_ms > $this->state->get('exact_online.api_rate_limits.minutelyLimitReset')) {
      $this->setApiRateLimits();
    }
    if ($time_in_ms > $this->state->get('exact_online.api_rate_limits.dailyLimitReset')) {
      $this->setApiRateLimits();
    }
    if ($this->state->get('exact_online.api_rate_limits.dailyLimitRemaining') === 0) {
      throw new \RuntimeException('Daily API rate limit is zero.');
    }
    if ($this->state->get('exact_online.api_rate_limits.minutelyLimitRemaining') === 0) {
      throw new \RuntimeException('Minute API rate limit is zero.');
    }
  }

  /**
   * Get and set API rate limits.
   *
   * Return all API rate limits info of current connection.
   * https://github.com/picqer/exact-php-client?tab=readme-ov-file#rate-limits.
   *
   * @return void
   *   Void.
   */
  protected function setApiRateLimits(): void {
    try {
      /*
       * @todo
       * We have to make a simple API call first for getting the API rate
       * limits from the response.
       * Otherwise, these values will be null.
       */
      // Retrieve your daily limit.
      $dailyLimit = $this->connection->getDailyLimit();
      // Retrieve the remaining amount of API calls for this day.
      $dailyLimitRemaining = $this->connection->getDailyLimitRemaining();
      // Retrieve the timestamp for when the limit will reset.
      $dailyLimitReset = $this->connection->getDailyLimitReset();
      // Retrieve your limit per minute.
      $minutelyLimit = $this->connection->getMinutelyLimit();
      // Retrieve the amount of API calls remaining for this minute.
      $minutelyLimitRemaining = $this->connection->getMinutelyLimitRemaining();
      // Retrieve the timestamp for when the minutely limit will reset.
      $minutelyLimitReset = $this->connection->getMinutelyLimitReset();
      if (is_null($dailyLimitRemaining)) {
        throw new \RuntimeException('Missing remaining daily limit');
      }
      if (is_null($minutelyLimitRemaining)) {
        throw new \RuntimeException('Missing remaining minutes limit');
      }
      // Put limit values in the state api.
      $this->state->set('exact_online.api_rate_limits.dailyLimit', $dailyLimit);
      $this->state->set('exact_online.api_rate_limits.dailyLimitRemaining', $dailyLimitRemaining);
      $this->state->set('exact_online.api_rate_limits.dailyLimitReset', $dailyLimitReset);
      $this->state->set('exact_online.api_rate_limits.minutelyLimit', $minutelyLimit);
      $this->state->set('exact_online.api_rate_limits.minutelyLimitRemaining', $minutelyLimitRemaining);
      $this->state->set('exact_online.api_rate_limits.minutelyLimitReset', $minutelyLimitReset);
    }
    catch (\Exception $e) {
      throw new \RuntimeException('Error getting API rate limits: ' . $e->getMessage());
    }
  }

  /**
   * Update and restore tokens from API.
   *
   * Every time this function is called, the tokens are refreshed if needed by
   * the exact-php-client library. A new access token expires in 10 minutes.
   * https://support.exactonline.com/community/s/knowledge-base#All-All-DNO-Content-oauth-eol-oauth-devstep3
   * You can only request a new access token 9 minutes and 30 seconds after you
   * received it. If you request an access token before the 9 minutes and 30
   * seconds have passed, your call will be rejected with response code 401 and
   * reason Access Token not expired.
   *
   * Refresh tokens are valid for 30 days.
   * If your refresh token has expired, you must obtain a new refresh token
   * through the authorization code grant type flow. If the app is not used in
   * 30 days, you must make a new authorization request to the user.
   *
   * @throws \Picqer\Financials\Exact\ApiException
   *   Exception.
   */
  protected function updateAndStoreTokens(): void {
    $tokenDetails = $this->getTokenExpirationDetails();
    if ($tokenDetails['status'] === 'expired') {
      try {
        $tokens_from_state = $this->state->get('exact_online.tokens');
        if ($this->connection->getRefreshToken() !== $tokens_from_state['refresh_token']) {
          $this->logInfo('Refresh token from state !== refresh token from connection');
        }
        $this->restoreTokens();
        $this->connection->connect();
        $reconnect = TRUE;
        $this->logInfo('Re-connected successfully.');
      }
      catch (\Exception $e) {
        if ($e->getCode() === 401) {
          $this->logInfo($this->connection->getRefreshToken() . ' -> this is the (old) refresh token from connection');
          // Delete tokens from state.
          $this->deleteTokens();
        }
        $this->logError('Failed to get a new access token, reason: ' . $e->getMessage());
        throw $e;
      }
    }
    // Save tokens with the state API.
    $tokens = [
      'access_token' => $this->connection->getAccessToken(),
      'refresh_token' => $this->connection->getRefreshToken(),
      'expires_in' => $this->connection->getTokenExpires(),
    ];
    $this->state->set('exact_online.tokens', $tokens);
    $reconnected = (isset($reconnect)) ? 'yes' : 'no';
    $this->logInfo('Tokens saved successfully (reconnect: ' . $reconnected . ')');
  }

  /**
   * Restores tokens from state to the connection.
   *
   * @return void
   *   Void.
   */
  protected function restoreTokens(): void {
    $tokens = $this->state->get('exact_online.tokens');

    if ($tokens) {
      try {
        if (!empty($tokens['access_token'])) {
          $this->connection->setAccessToken($tokens['access_token']);
        }
        if (!empty($tokens['refresh_token'])) {
          $this->connection->setRefreshToken($tokens['refresh_token']);
        }
        if (!empty($tokens['expires_in'])) {
          $this->connection->setTokenExpires($tokens['expires_in']);
        }
      }
      catch (\Exception $e) {
        $this->logError('Failed to restore tokens to the connection, reason: ' . $e->getMessage());
      }
    }
  }

  /**
   * Delete tokens.
   *
   * @return void
   *   Void.
   */
  protected function deleteTokens(): void {
    $tokens = $this->state->get('exact_online.tokens');

    if ($tokens) {
      try {
        if (!empty($tokens['access_token'])) {
          $this->connection->setAccessToken(NULL);
        }
        if (!empty($tokens['refresh_token'])) {
          $this->connection->setRefreshToken(NULL);
        }
        if (!empty($tokens['expires_in'])) {
          $this->connection->setTokenExpires(0);
        }
        $this->state->delete('exact_online.tokens');
      }
      catch (\Exception $e) {
        $this->logError('Failed to delete tokens: ' . $e->getMessage());
      }
    }
  }

  /**
   * Clear API rate limits.
   *
   * @return void
   *   Void.
   */
  protected function clearApiRateLimits(): void {
    $apiRateLimits = $this->state->get('exact_online.api_rate_limits');
    if ($apiRateLimits) {
      try {
        $this->state->delete('exact_online.api_rate_limits');
      }
      catch (\Exception $e) {
        $this->logError('Failed to clear api rate limits values from state: ' . $e->getMessage());
      }
    }
  }

  /**
   * Logs an error message.
   *
   * @param string $message
   *   The message to log.
   */
  protected function logError(string $message): void {
    $this->loggerFactory->error($message);
    $this->addLogEntry('error', $message);
  }

  /**
   * Logs an info message.
   *
   * @param string $message
   *   The message to log.
   */
  protected function logInfo(string $message): void {
    $this->loggerFactory->info($message);
    $this->addLogEntry('info', $message);
  }

  /**
   * Adds a log entry to the state storage.
   *
   * @param string $type
   *   The type of log entry.
   * @param string $message
   *   The log message.
   */
  protected function addLogEntry(string $type, string $message): void {
    $logs = $this->state->get('exact_online.logs', []);

    $logs[] = [
      'timestamp' => date('Y-m-d H:i:s'),
      'type' => $type,
      'message' => $message,
      'user' => $this->user->id(),
    ];

    // Keep only the last 1000 log entries.
    if (count($logs) > 1000) {
      array_shift($logs);
    }

    $this->state->set('exact_online.logs', $logs);
  }

  /**
   * Gets a human-readable message about token expiration status.
   *
   * @param string $status
   *   The token status.
   * @param float|null $remainingMinutes
   *   The remaining hours until expiration.
   *
   * @return string
   *   A human-readable message about token expiration.
   */
  protected function getExpirationMessage(string $status, ?float $remainingMinutes): string {
    switch ($status) {
      case 'expired':
        return $this->t('Token has expired');

      case 'expiring_soon':
        return $this->t('Token will expire in @minutes minutes', [
          '@minutes' => $remainingMinutes,
        ]);

      case 'valid':
        return $this->t('Token is valid for @minutes minutes', [
          '@minutes' => $remainingMinutes,
        ]);

      default:
        return $this->t('Token status unknown');
    }
  }

}
