<?php

namespace Drupal\exact_online\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Service for handling Exact Online token expiration notifications.
 */
class TokenExpirationNotifier {

  use StringTranslationTrait;

  /**
   * The Exact Online service.
   *
   * @var \Drupal\exact_online\Service\ExactOnlineServiceInterface
   */
  protected $exactOnlineService;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new TokenExpirationNotifier.
   */
  public function __construct(
    ExactOnlineServiceInterface $exact_online_service,
    MailManagerInterface $mail_manager,
    StateInterface $state,
    TranslationInterface $string_translation,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->exactOnlineService = $exact_online_service;
    $this->mailManager = $mail_manager;
    $this->state = $state;
    $this->stringTranslation = $string_translation;
    $this->configFactory = $config_factory;
  }

  /**
   * Checks token expiration and sends notifications if needed.
   */
  public function checkAndNotify(): void {
    $connection = $this->exactOnlineService->getConnection();
    if ($connection->getRefreshToken()) {
      $details = $this->exactOnlineService->getTokenExpirationDetails();
      if ($details['status'] === 'expiring_soon') {
        // Check if we've already notified for this expiration.
        $lastNotification = $this->state->get('exact_online.last_expiration_notification', 0);

        // Only notify once every 24 hours.
        if ((time() - $lastNotification) >= 86400) {
          $this->sendExpirationNotification($details);
          $this->state->set('exact_online.last_expiration_notification', time());
        }
      }
    }
    else {
      $details = [
        'expires_at' => 'already expired / no refresh token set',
        'remaining_hours' => 0,
      ];
      $this->sendExpirationNotification($details);
      $this->state->set('exact_online.last_expiration_notification', time());
    }
  }

  /**
   * Sends a token expiration notification.
   *
   * @param array $details
   *   Token expiration details.
   */
  protected function sendExpirationNotification(array $details): void {
    $site = $this->configFactory->get('system.site');
    $to = $site->get('mail');

    $params = [
      'subject' => $this->t('Exact Online token warning'),
      'body' => [
        $this->t('The Exact Online access token will expire soon:'),
        $this->t('Expiration time: @time', ['@time' => $details['expires_at']]),
        $this->t('Remaining hours: @hours', ['@hours' => $details['remaining_hours']]),
        $this->t('Please take action to refresh the token.'),
      ],
    ];

    $this->mailManager->mail(
      'exact_online',
      'token_expiration',
      $to,
      'en',
      $params
    );
  }

}
