<?php

namespace Drupal\exchangerate\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\HttpFoundation\Request;
use Drupal\exchangerate\Service\ExchangeRateApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for handling currency autocomplete.
 */
class CurrencyController extends ControllerBase {
  use StringTranslationTrait;

  /**
   * The exchange rate API service.
   *
   * @var \Drupal\exchangerate\Service\ExchangeRateApi
   */
  protected $apiService;

  /**
   * Constructs a new CurrencyController.
   *
   * @param \Drupal\exchangerate\Service\ExchangeRateApi $api_service
   *   The exchange rate API service.
   */
  public function __construct(ExchangeRateApi $api_service) {
    $this->apiService = $api_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('exchangerate.api')
    );
  }

  /**
   * Autocomplete callback for single currency.
   */
  public function autocomplete(Request $request) {
    // Get the query parameter 'string' from the URL.
    $string = $request->query->get('q');

    // If the string is empty, return an empty array.
    if (empty($string)) {
      return new JsonResponse([]);
    }

    $currencies = $this->apiService->getCountries();

    $matches = [];
    // Normalize the search string to lowercase.
    $search_string = strtolower($string);

    foreach ($currencies as $currency_code => $currency) {
      // Convert the currency name and code to lowercase for comparison.
      $currency_name_lower = strtolower($currency['name']);
      $currency_code_lower = strtolower($currency_code);

      // Check if the lowercase version of either the currency name
      // or code contains the search string.
      if (strpos($currency_name_lower, $search_string) !== FALSE || strpos($currency_code_lower, $search_string) !== FALSE) {
        $matches[] = [
          'value' => $currency_code,
          'label' => $currency['name'] . ' (' . $currency_code . ' ' . $currency['flag_emoji'] . ')',
          'flag_emoji' => $currency['flag_emoji'],
        ];
      }
    }

    return new JsonResponse($matches);
  }

  /**
   * Autocomplete callback for multiple currencies.
   */
  public function autocompleteMultiple(Request $request) {
    return $this->autocomplete($request);
  }

}
