<?php

namespace Drupal\exchangerate\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Exchange Rate settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The exchange rate API service.
   *
   * @var \Drupal\exchangerate\Service\ExchangeRateApi
   */
  protected $apiService;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'exchangerate_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['exchangerate.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // Call the parent create method to inject ConfigFactoryInterface
    // and other dependencies.
    $form = parent::create($container);

    // Now inject the ExchangeRateApi service.
    $form->apiService = $container->get('exchangerate.api');
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('exchangerate.settings');

    $form['exchangerate_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter your API key from ExchangeRate-API.'),
      '#default_value' => $config->get('api_key'),
      '#required' => TRUE,
    ];

    // Fetch countries using the API service.
    $countries = $this->apiService->getCountries();
    $country_options = [];
    foreach ($countries as $currency_code => $country) {
      $country_options[$currency_code] = $country['name'] . $country['flag_emoji'] . ' (' . $currency_code . ')';
    }

    // Add base currency selection form element.
    $form['exchangerate_base_currency'] = [
      '#type' => 'select',
      '#title' => $this->t('Base Currency'),
      '#description' => $this->t('Select the base currency for exchange rates.'),
      '#options' => $country_options,
      '#default_value' => $config->get('base_currency') ?: 'USD',
    ];

    // Cache settings form elements.
    $form['exchangerate_cache'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Cache Settings'),
    ];

    $form['exchangerate_cache']['exchangerate_cache_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable API data caching'),
      '#description' => $this->t('Store API response data in cache to reduce API calls.'),
      '#default_value' => $config->get('cache_enabled'),
    ];

    $form['exchangerate_cache']['exchangerate_cache_time'] = [
      '#type' => 'number',
      '#min' => 1,
      '#title' => $this->t('Cache Duration'),
      '#description' => $this->t('How long to store the data in the cache.'),
      '#default_value' => $config->get('cache_time') ?: 1,
      '#states' => [
        'visible' => [
          ':input[name="exchangerate_cache_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['exchangerate_cache']['exchangerate_cache_unit'] = [
      '#type' => 'select',
      '#title' => $this->t('Cache Unit'),
      '#options' => [
        'minutes' => $this->t('Minutes'),
        'hours' => $this->t('Hours'),
      ],
      '#default_value' => $config->get('cache_unit') ?: 'hours',
      '#states' => [
        'visible' => [
          ':input[name="exchangerate_cache_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate the cache time.
    $this->validateCacheDuration($form_state);
  }

  /**
   * Custom validation for cache duration.
   */
  protected function validateCacheDuration(FormStateInterface $form_state) {
    $cache_time = $form_state->getValue('exchangerate_cache_time');
    $cache_unit = $form_state->getValue('exchangerate_cache_unit');

    // Validate cache time does not exceed one day (1440 minutes or 24 hours).
    if ($cache_unit == 'minutes' && $cache_time > 1440) {
      $form_state->setErrorByName('exchangerate_cache_time', $this->t('Cache duration in minutes cannot exceed 1440 minutes (1 day).'));
    }
    elseif ($cache_unit == 'hours' && $cache_time > 24) {
      $form_state->setErrorByName('exchangerate_cache_time', $this->t('Cache duration in hours cannot exceed 24 hours (1 day).'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('exchangerate.settings')
      ->set('api_key', $form_state->getValue('exchangerate_api_key'))
      ->set('base_currency', $form_state->getValue('exchangerate_base_currency'))
      ->set('cache_enabled', $form_state->getValue('exchangerate_cache_enabled'))
      ->set('cache_time', $form_state->getValue('exchangerate_cache_time'))
      ->set('cache_unit', $form_state->getValue('exchangerate_cache_unit'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
