<?php

namespace Drupal\exchangerate\Service;

use GuzzleHttp\Exception\ConnectException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Service for fetching and caching exchange rate data.
 */
class ExchangeRateApi {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private $configFactory;

  /**
   * The HTTP client factory.
   *
   * @var \Drupal\Core\Http\ClientFactory
   */
  protected $httpClientFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The module handler service.
   *
   * Used to invoke hook_alter() and check module existence.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * ExchangeRateApi constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Http\ClientFactory $http_client_factory
   *   The HTTP client factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service, used to translate human-readable text
   *   in the service, like country names, into different languages.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ClientFactory $http_client_factory, CacheBackendInterface $cache_backend, LoggerChannelFactoryInterface $logger_factory, ModuleHandlerInterface $module_handler, TranslationInterface $string_translation) {
    $this->configFactory = $config_factory;
    $this->httpClientFactory = $http_client_factory;
    $this->cache = $cache_backend;
    $this->logger = $logger_factory->get('exchangerate');
    $this->moduleHandler = $module_handler;
    $this->stringTranslation = $string_translation;
  }

  /**
   * Fetches exchange rates from the API, using cache if available.
   *
   * @return array
   *   An array of exchange rates or an empty array on failure.
   */
  public function getRates() {
    $config = $this->configFactory->get('exchangerate.settings');
    $apiKey = $config->get('api_key');
    $baseCurrency = $config->get('base_currency') ?: 'USD';
    $cacheEnabled = $config->get('cache_enabled');

    if (!$apiKey) {
      return [];
    }

    $cache_id = 'exchangerate_rates_' . $baseCurrency;
    $cache_entry = $this->cache->get($cache_id);

    // If cache exists, check date validity.
    if ($cacheEnabled && $cache_entry) {
      $cached_data = $cache_entry->data;

      // Validate if the cached date matches today.
      if (isset($cached_data['date']) && $cached_data['date'] === date('Y-m-d')) {
        return $cached_data['rates'];
      }
    }

    $exchangeRateUrl = "https://v6.exchangerate-api.com/v6/{$apiKey}/latest/{$baseCurrency}";
    $httpClient = $this->httpClientFactory->fromOptions(['verify' => FALSE]);

    try {
      $response = $httpClient->get($exchangeRateUrl);
      $data = json_decode($response->getBody(), TRUE);

      if (isset($data['conversion_rates'])) {
        $rates = $data['conversion_rates'];

        // Store rates + cache date.
        $cache_data = [
          'date' => date('Y-m-d'),
          'rates' => $rates,
        ];

        // Determine cache expiration (e.g., until tomorrow midnight)
        $expire = 0;
        if ($cacheEnabled) {
          $expire = $this->getCacheTimestamp();
        }

        $this->cache->set($cache_id, $cache_data, $expire);

        return $rates;
      }
    }
    catch (ConnectException $e) {
      $this->logger->error('Connection error: @message', ['@message' => $e->getMessage()]);
    }
    catch (RequestException $e) {
      $this->logger->error('API request failed: @message', ['@message' => $e->getMessage()]);
    }

    return [];
  }

  /**
   * Retrieves the base currency from the configuration.
   *
   * @return string
   *   The base currency.
   */
  public function getBaseCurrency() {
    return $this->configFactory->get('exchangerate.settings')->get('base_currency');
  }

  /**
   * Retrieves the cache enabled configuration value.
   *
   * @return bool
   *   TRUE if cache is enabled, FALSE otherwise.
   */
  public function isCacheEnabled() {
    return $this->configFactory->get('exchangerate.settings')->get('cache_enabled');
  }

  /**
   * Retrieves the cache duration timestamp with unit conversion.
   *
   * @return int
   *   The cache timestamp in seconds.
   */
  public function getCacheTimestamp() {
    $config = $this->configFactory->get('exchangerate.settings');

    // Retrieve the cache time and unit settings.
    $cache_time = $config->get('cache_time') ?: 1;
    $cache_unit = $config->get('cache_unit') ?: 'hours';

    // Convert the time based on the unit.
    switch ($cache_unit) {
      case 'minutes':
        // Convert minutes to seconds.
        $cache_duration_seconds = $cache_time * 60;
        break;

      case 'hours':
        // Convert hours to seconds.
        $cache_duration_seconds = $cache_time * 60 * 60;
        break;

      default:
        // Default to hours if an unknown unit is found.
        $cache_duration_seconds = $cache_time * 60 * 60;
        break;
    }

    // Return the timestamp (current time + cache duration)
    return time() + $cache_duration_seconds;
  }

  /**
   * Retrieves a list of countries and their currency data.
   *
   * @return array
   *   An array of countries.
   */
  public function getCountries() {
    // Define the comprehensive list of countries with their currency
    // data and flag emojis, using currency_code as the key.
    $countries = [
      "AFN" => [
        "name" => $this->stringTranslation->translate("Afghanistan"),
        "currency_code" => "AFN",
        "currency_symbol" => "؋",
        "flag_emoji" => "🇦🇫",
      ],
      "ALL" => [
        "name" => $this->stringTranslation->translate("Albania"),
        "currency_code" => "ALL",
        "currency_symbol" => "Lek",
        "flag_emoji" => "🇦🇱",
      ],
      "DZD" => [
        "name" => $this->stringTranslation->translate("Algeria"),
        "currency_code" => "DZD",
        "currency_symbol" => "د.ج",
        "flag_emoji" => "🇩🇿",
      ],
      "EUR" => [
        "name" => $this->stringTranslation->translate("Andorra"),
        "currency_code" => "EUR",
        "currency_symbol" => "€",
        "flag_emoji" => "🇦🇩",
      ],
      "AOA" => [
        "name" => $this->stringTranslation->translate("Angola"),
        "currency_code" => "AOA",
        "currency_symbol" => "Kz",
        "flag_emoji" => "🇦🇴",
      ],
      "XCD" => [
        "name" => $this->stringTranslation->translate("Antigua and Barbuda"),
        "currency_code" => "XCD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇦🇬",
      ],
      "ARS" => [
        "name" => $this->stringTranslation->translate("Argentina"),
        "currency_code" => "ARS",
        "currency_symbol" => "$",
        "flag_emoji" => "🇦🇷",
      ],
      "AMD" => [
        "name" => $this->stringTranslation->translate("Armenia"),
        "currency_code" => "AMD",
        "currency_symbol" => "֏",
        "flag_emoji" => "🇦🇲",
      ],
      "AUD" => [
        "name" => $this->stringTranslation->translate("Australia"),
        "currency_code" => "AUD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇦🇺",
      ],
      "AZN" => [
        "name" => $this->stringTranslation->translate("Azerbaijan"),
        "currency_code" => "AZN",
        "currency_symbol" => "₼",
        "flag_emoji" => "🇦🇿",
      ],
      "BSD" => [
        "name" => $this->stringTranslation->translate("Bahamas"),
        "currency_code" => "BSD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇧🇸",
      ],
      "BHD" => [
        "name" => $this->stringTranslation->translate("Bahrain"),
        "currency_code" => "BHD",
        "currency_symbol" => ".د.ب",
        "flag_emoji" => "🇧🇭",
      ],
      "BDT" => [
        "name" => $this->stringTranslation->translate("Bangladesh"),
        "currency_code" => "BDT",
        "currency_symbol" => "৳",
        "flag_emoji" => "🇧🇩",
      ],
      "BBD" => [
        "name" => $this->stringTranslation->translate("Barbados"),
        "currency_code" => "BBD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇧🇧",
      ],
      "BYN" => [
        "name" => $this->stringTranslation->translate("Belarus"),
        "currency_code" => "BYN",
        "currency_symbol" => "Br",
        "flag_emoji" => "🇧🇾",
      ],
      "BZD" => [
        "name" => $this->stringTranslation->translate("Belize"),
        "currency_code" => "BZD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇧🇿",
      ],
      "XOF" => [
        "name" => $this->stringTranslation->translate("Benin"),
        "currency_code" => "XOF",
        "currency_symbol" => "Fr",
        "flag_emoji" => "🇧🇯",
      ],
      "BTN" => [
        "name" => $this->stringTranslation->translate("Bhutan"),
        "currency_code" => "BTN",
        "currency_symbol" => "Nu.",
        "flag_emoji" => "🇧🇹",
      ],
      "BOB" => [
        "name" => $this->stringTranslation->translate("Bolivia"),
        "currency_code" => "BOB",
        "currency_symbol" => "Bs.",
        "flag_emoji" => "🇧🇴",
      ],
      "BAM" => [
        "name" => $this->stringTranslation->translate("Bosnia and Herzegovina"),
        "currency_code" => "BAM",
        "currency_symbol" => "KM",
        "flag_emoji" => "🇧🇦",
      ],
      "BWP" => [
        "name" => $this->stringTranslation->translate("Botswana"),
        "currency_code" => "BWP",
        "currency_symbol" => "P",
        "flag_emoji" => "🇧🇼",
      ],
      "BRL" => [
        "name" => $this->stringTranslation->translate("Brazil"),
        "currency_code" => "BRL",
        "currency_symbol" => "R$",
        "flag_emoji" => "🇧🇷",
      ],
      "BND" => [
        "name" => $this->stringTranslation->translate("Brunei"),
        "currency_code" => "BND",
        "currency_symbol" => "$",
        "flag_emoji" => "🇧🇳",
      ],
      "BGN" => [
        "name" => $this->stringTranslation->translate("Bulgaria"),
        "currency_code" => "BGN",
        "currency_symbol" => "лв",
        "flag_emoji" => "🇧🇬",
      ],
      "BIF" => [
        "name" => $this->stringTranslation->translate("Burundi"),
        "currency_code" => "BIF",
        "currency_symbol" => "FBu",
        "flag_emoji" => "🇧🇮",
      ],
      "CVE" => [
        "name" => $this->stringTranslation->translate("Cabo Verde"),
        "currency_code" => "CVE",
        "currency_symbol" => "$",
        "flag_emoji" => "🇨🇻",
      ],
      "KHR" => [
        "name" => $this->stringTranslation->translate("Cambodia"),
        "currency_code" => "KHR",
        "currency_symbol" => "៛",
        "flag_emoji" => "🇰🇭",
      ],
      "XAF" => [
        "name" => $this->stringTranslation->translate("Cameroon"),
        "currency_code" => "XAF",
        "currency_symbol" => "Fr",
        "flag_emoji" => "🇨🇲",
      ],
      "CAD" => [
        "name" => $this->stringTranslation->translate("Canada"),
        "currency_code" => "CAD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇨🇦",
      ],
      "CLP" => [
        "name" => $this->stringTranslation->translate("Chile"),
        "currency_code" => "CLP",
        "currency_symbol" => "$",
        "flag_emoji" => "🇨🇱",
      ],
      "CNY" => [
        "name" => $this->stringTranslation->translate("China"),
        "currency_code" => "CNY",
        "currency_symbol" => "¥",
        "flag_emoji" => "🇨🇳",
      ],
      "COP" => [
        "name" => $this->stringTranslation->translate("Colombia"),
        "currency_code" => "COP",
        "currency_symbol" => "$",
        "flag_emoji" => "🇨🇴",
      ],
      "KMF" => [
        "name" => $this->stringTranslation->translate("Comoros"),
        "currency_code" => "KMF",
        "currency_symbol" => "CF",
        "flag_emoji" => "🇰🇲",
      ],
      "CRC" => [
        "name" => $this->stringTranslation->translate("Costa Rica"),
        "currency_code" => "CRC",
        "currency_symbol" => "₡",
        "flag_emoji" => "🇨🇷",
      ],
      "CUP" => [
        "name" => $this->stringTranslation->translate("Cuba"),
        "currency_code" => "CUP",
        "currency_symbol" => "₱",
        "flag_emoji" => "🇨🇺",
      ],
      "CZK" => [
        "name" => $this->stringTranslation->translate("Czech Republic"),
        "currency_code" => "CZK",
        "currency_symbol" => "Kč",
        "flag_emoji" => "🇨🇿",
      ],
      "CDF" => [
        "name" => $this->stringTranslation->translate("Democratic Republic of the Congo"),
        "currency_code" => "CDF",
        "currency_symbol" => "FC",
        "flag_emoji" => "🇨🇩",
      ],
      "DKK" => [
        "name" => $this->stringTranslation->translate("Denmark"),
        "currency_code" => "DKK",
        "currency_symbol" => "kr",
        "flag_emoji" => "🇩🇰",
      ],
      "DJF" => [
        "name" => $this->stringTranslation->translate("Djibouti"),
        "currency_code" => "DJF",
        "currency_symbol" => "Fdj",
        "flag_emoji" => "🇩🇯",
      ],
      "DOP" => [
        "name" => $this->stringTranslation->translate("Dominican Republic"),
        "currency_code" => "DOP",
        "currency_symbol" => "RD$",
        "flag_emoji" => "🇩🇴",
      ],
      "USD" => [
        "name" => $this->stringTranslation->translate("United States of America"),
        "currency_code" => "USD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇺🇸",
      ],
      "EGP" => [
        "name" => $this->stringTranslation->translate("Egypt"),
        "currency_code" => "EGP",
        "currency_symbol" => "£",
        "flag_emoji" => "🇪🇬",
      ],
      "ERN" => [
        "name" => $this->stringTranslation->translate("Eritrea"),
        "currency_code" => "ERN",
        "currency_symbol" => "Nfk",
        "flag_emoji" => "🇪🇷",
      ],
      "SZL" => [
        "name" => $this->stringTranslation->translate("Eswatini"),
        "currency_code" => "SZL",
        "currency_symbol" => "L",
        "flag_emoji" => "🇸🇿",
      ],
      "ETB" => [
        "name" => $this->stringTranslation->translate("Ethiopia"),
        "currency_code" => "ETB",
        "currency_symbol" => "Br",
        "flag_emoji" => "🇪🇹",
      ],
      "FJD" => [
        "name" => $this->stringTranslation->translate("Fiji"),
        "currency_code" => "FJD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇫🇯",
      ],
      "GMD" => [
        "name" => $this->stringTranslation->translate("Gambia"),
        "currency_code" => "GMD",
        "currency_symbol" => "D",
        "flag_emoji" => "🇬🇲",
      ],
      "GEL" => [
        "name" => $this->stringTranslation->translate("Georgia"),
        "currency_code" => "GEL",
        "currency_symbol" => "₾",
        "flag_emoji" => "🇬🇪",
      ],
      "GHS" => [
        "name" => $this->stringTranslation->translate("Ghana"),
        "currency_code" => "GHS",
        "currency_symbol" => "₵",
        "flag_emoji" => "🇬🇭",
      ],
      "GTQ" => [
        "name" => $this->stringTranslation->translate("Guatemala"),
        "currency_code" => "GTQ",
        "currency_symbol" => "Q",
        "flag_emoji" => "🇬🇹",
      ],
      "GNF" => [
        "name" => $this->stringTranslation->translate("Guinea"),
        "currency_code" => "GNF",
        "currency_symbol" => "Fr",
        "flag_emoji" => "🇬🇳",
      ],
      "GYD" => [
        "name" => $this->stringTranslation->translate("Guyana"),
        "currency_code" => "GYD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇬🇾",
      ],
      "HTG" => [
        "name" => $this->stringTranslation->translate("Haiti"),
        "currency_code" => "HTG",
        "currency_symbol" => "G",
        "flag_emoji" => "🇭🇹",
      ],
      "HNL" => [
        "name" => $this->stringTranslation->translate("Honduras"),
        "currency_code" => "HNL",
        "currency_symbol" => "L",
        "flag_emoji" => "🇭🇳",
      ],
      "HUF" => [
        "name" => $this->stringTranslation->translate("Hungary"),
        "currency_code" => "HUF",
        "currency_symbol" => "Ft",
        "flag_emoji" => "🇭🇺",
      ],
      "ISK" => [
        "name" => $this->stringTranslation->translate("Iceland"),
        "currency_code" => "ISK",
        "currency_symbol" => "kr",
        "flag_emoji" => "🇮🇸",
      ],
      "INR" => [
        "name" => $this->stringTranslation->translate("India"),
        "currency_code" => "INR",
        "currency_symbol" => "₹",
        "flag_emoji" => "🇮🇳",
      ],
      "IDR" => [
        "name" => $this->stringTranslation->translate("Indonesia"),
        "currency_code" => "IDR",
        "currency_symbol" => "Rp",
        "flag_emoji" => "🇮🇩",
      ],
      "IRR" => [
        "name" => $this->stringTranslation->translate("Iran"),
        "currency_code" => "IRR",
        "currency_symbol" => "﷼",
        "flag_emoji" => "🇮🇷",
      ],
      "IQD" => [
        "name" => $this->stringTranslation->translate("Iraq"),
        "currency_code" => "IQD",
        "currency_symbol" => "ع.د",
        "flag_emoji" => "🇮🇶",
      ],
      "ILS" => [
        "name" => $this->stringTranslation->translate("Israel"),
        "currency_code" => "ILS",
        "currency_symbol" => "₪",
        "flag_emoji" => "🇮🇱",
      ],
      "JMD" => [
        "name" => $this->stringTranslation->translate("Jamaica"),
        "currency_code" => "JMD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇯🇲",
      ],
      "JPY" => [
        "name" => $this->stringTranslation->translate("Japan"),
        "currency_code" => "JPY",
        "currency_symbol" => "¥",
        "flag_emoji" => "🇯🇵",
      ],
      "JOD" => [
        "name" => $this->stringTranslation->translate("Jordan"),
        "currency_code" => "JOD",
        "currency_symbol" => "د.ا",
        "flag_emoji" => "🇯🇴",
      ],
      "KZT" => [
        "name" => $this->stringTranslation->translate("Kazakhstan"),
        "currency_code" => "KZT",
        "currency_symbol" => "₸",
        "flag_emoji" => "🇰🇿",
      ],
      "KES" => [
        "name" => $this->stringTranslation->translate("Kenya"),
        "currency_code" => "KES",
        "currency_symbol" => "Sh",
        "flag_emoji" => "🇰🇪",
      ],
      "KWD" => [
        "name" => $this->stringTranslation->translate("Kuwait"),
        "currency_code" => "KWD",
        "currency_symbol" => "د.ك",
        "flag_emoji" => "🇰🇼",
      ],
      "KGS" => [
        "name" => $this->stringTranslation->translate("Kyrgyzstan"),
        "currency_code" => "KGS",
        "currency_symbol" => "с",
        "flag_emoji" => "🇰🇬",
      ],
      "LAK" => [
        "name" => $this->stringTranslation->translate("Laos"),
        "currency_code" => "LAK",
        "currency_symbol" => "₭",
        "flag_emoji" => "🇱🇦",
      ],
      "LBP" => [
        "name" => $this->stringTranslation->translate("Lebanon"),
        "currency_code" => "LBP",
        "currency_symbol" => "ل.ل",
        "flag_emoji" => "🇱🇧",
      ],
      "LSL" => [
        "name" => $this->stringTranslation->translate("Lesotho"),
        "currency_code" => "LSL",
        "currency_symbol" => "L",
        "flag_emoji" => "🇱🇸",
      ],
      "LRD" => [
        "name" => $this->stringTranslation->translate("Liberia"),
        "currency_code" => "LRD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇱🇷",
      ],
      "LYD" => [
        "name" => $this->stringTranslation->translate("Libya"),
        "currency_code" => "LYD",
        "currency_symbol" => "ل.د",
        "flag_emoji" => "🇱🇾",
      ],
      "CHF" => [
        "name" => $this->stringTranslation->translate("Switzerland"),
        "currency_code" => "CHF",
        "currency_symbol" => "Fr",
        "flag_emoji" => "🇨🇭",
      ],
      "MGA" => [
        "name" => $this->stringTranslation->translate("Madagascar"),
        "currency_code" => "MGA",
        "currency_symbol" => "Ar",
        "flag_emoji" => "🇲🇬",
      ],
      "MWK" => [
        "name" => $this->stringTranslation->translate("Malawi"),
        "currency_code" => "MWK",
        "currency_symbol" => "K",
        "flag_emoji" => "🇲🇼",
      ],
      "MYR" => [
        "name" => $this->stringTranslation->translate("Malaysia"),
        "currency_code" => "MYR",
        "currency_symbol" => "RM",
        "flag_emoji" => "🇲🇾",
      ],
      "MVR" => [
        "name" => $this->stringTranslation->translate("Maldives"),
        "currency_code" => "MVR",
        "currency_symbol" => "Rf",
        "flag_emoji" => "🇲🇻",
      ],
      "MRU" => [
        "name" => $this->stringTranslation->translate("Mauritania"),
        "currency_code" => "MRU",
        "currency_symbol" => "UM",
        "flag_emoji" => "🇲🇷",
      ],
      "MUR" => [
        "name" => $this->stringTranslation->translate("Mauritius"),
        "currency_code" => "MUR",
        "currency_symbol" => "₨",
        "flag_emoji" => "🇲🇺",
      ],
      "MXN" => [
        "name" => $this->stringTranslation->translate("Mexico"),
        "currency_code" => "MXN",
        "currency_symbol" => "$",
        "flag_emoji" => "🇲🇽",
      ],
      "MDL" => [
        "name" => $this->stringTranslation->translate("Moldova"),
        "currency_code" => "MDL",
        "currency_symbol" => "L",
        "flag_emoji" => "🇲🇩",
      ],
      "MNT" => [
        "name" => $this->stringTranslation->translate("Mongolia"),
        "currency_code" => "MNT",
        "currency_symbol" => "₮",
        "flag_emoji" => "🇲🇳",
      ],
      "MAD" => [
        "name" => $this->stringTranslation->translate("Morocco"),
        "currency_code" => "MAD",
        "currency_symbol" => "د.م.",
        "flag_emoji" => "🇲🇦",
      ],
      "MZN" => [
        "name" => $this->stringTranslation->translate("Mozambique"),
        "currency_code" => "MZN",
        "currency_symbol" => "MT",
        "flag_emoji" => "🇲🇿",
      ],
      "MMK" => [
        "name" => $this->stringTranslation->translate("Myanmar"),
        "currency_code" => "MMK",
        "currency_symbol" => "Ks",
        "flag_emoji" => "🇲🇲",
      ],
      "NAD" => [
        "name" => $this->stringTranslation->translate("Namibia"),
        "currency_code" => "NAD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇳🇦",
      ],
      "NPR" => [
        "name" => $this->stringTranslation->translate("Nepal"),
        "currency_code" => "NPR",
        "currency_symbol" => "₨",
        "flag_emoji" => "🇳🇵",
      ],
      "NZD" => [
        "name" => $this->stringTranslation->translate("New Zealand"),
        "currency_code" => "NZD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇳🇿",
      ],
      "NIO" => [
        "name" => $this->stringTranslation->translate("Nicaragua"),
        "currency_code" => "NIO",
        "currency_symbol" => "C$",
        "flag_emoji" => "🇳🇮",
      ],
      "NGN" => [
        "name" => $this->stringTranslation->translate("Nigeria"),
        "currency_code" => "NGN",
        "currency_symbol" => "₦",
        "flag_emoji" => "🇳🇬",
      ],
      "KPW" => [
        "name" => $this->stringTranslation->translate("North Korea"),
        "currency_code" => "KPW",
        "currency_symbol" => "₩",
        "flag_emoji" => "🇰🇵",
      ],
      "MKD" => [
        "name" => $this->stringTranslation->translate("North Macedonia"),
        "currency_code" => "MKD",
        "currency_symbol" => "ден",
        "flag_emoji" => "🇲🇰",
      ],
      "NOK" => [
        "name" => $this->stringTranslation->translate("Norway"),
        "currency_code" => "NOK",
        "currency_symbol" => "kr",
        "flag_emoji" => "🇳🇴",
      ],
      "OMR" => [
        "name" => $this->stringTranslation->translate("Oman"),
        "currency_code" => "OMR",
        "currency_symbol" => "ر.ع.",
        "flag_emoji" => "🇴🇲",
      ],
      "PKR" => [
        "name" => $this->stringTranslation->translate("Pakistan"),
        "currency_code" => "PKR",
        "currency_symbol" => "₨",
        "flag_emoji" => "🇵🇰",
      ],
      "PAB" => [
        "name" => $this->stringTranslation->translate("Panama"),
        "currency_code" => "PAB",
        "currency_symbol" => "B/.",
        "flag_emoji" => "🇵🇦",
      ],
      "PGK" => [
        "name" => $this->stringTranslation->translate("Papua New Guinea"),
        "currency_code" => "PGK",
        "currency_symbol" => "K",
        "flag_emoji" => "🇵🇬",
      ],
      "PYG" => [
        "name" => $this->stringTranslation->translate("Paraguay"),
        "currency_code" => "PYG",
        "currency_symbol" => "₲",
        "flag_emoji" => "🇵🇾",
      ],
      "PEN" => [
        "name" => $this->stringTranslation->translate("Peru"),
        "currency_code" => "PEN",
        "currency_symbol" => "S/.",
        "flag_emoji" => "🇵🇪",
      ],
      "PHP" => [
        "name" => $this->stringTranslation->translate("Philippines"),
        "currency_code" => "PHP",
        "currency_symbol" => "₱",
        "flag_emoji" => "🇵🇭",
      ],
      "PLN" => [
        "name" => $this->stringTranslation->translate("Poland"),
        "currency_code" => "PLN",
        "currency_symbol" => "zł",
        "flag_emoji" => "🇵🇱",
      ],
      "QAR" => [
        "name" => $this->stringTranslation->translate("Qatar"),
        "currency_code" => "QAR",
        "currency_symbol" => "ر.ق",
        "flag_emoji" => "🇶🇦",
      ],
      "RON" => [
        "name" => $this->stringTranslation->translate("Romania"),
        "currency_code" => "RON",
        "currency_symbol" => "lei",
        "flag_emoji" => "🇷🇴",
      ],
      "RUB" => [
        "name" => $this->stringTranslation->translate("Russia"),
        "currency_code" => "RUB",
        "currency_symbol" => "₽",
        "flag_emoji" => "🇷🇺",
      ],
      "RWF" => [
        "name" => $this->stringTranslation->translate("Rwanda"),
        "currency_code" => "RWF",
        "currency_symbol" => "Fr",
        "flag_emoji" => "🇷🇼",
      ],
      "WST" => [
        "name" => $this->stringTranslation->translate("Samoa"),
        "currency_code" => "WST",
        "currency_symbol" => "T",
        "flag_emoji" => "🇼🇸",
      ],
      "STN" => [
        "name" => $this->stringTranslation->translate("Sao Tome and Principe"),
        "currency_code" => "STN",
        "currency_symbol" => "Db",
        "flag_emoji" => "🇸🇹",
      ],
      "SAR" => [
        "name" => $this->stringTranslation->translate("Saudi Arabia"),
        "currency_code" => "SAR",
        "currency_symbol" => "﷼",
        "flag_emoji" => "🇸🇦",
      ],
      "RSD" => [
        "name" => $this->stringTranslation->translate("Serbia"),
        "currency_code" => "RSD",
        "currency_symbol" => "дин",
        "flag_emoji" => "🇷🇸",
      ],
      "SCR" => [
        "name" => $this->stringTranslation->translate("Seychelles"),
        "currency_code" => "SCR",
        "currency_symbol" => "₨",
        "flag_emoji" => "🇸🇨",
      ],
      "SLL" => [
        "name" => $this->stringTranslation->translate("Sierra Leone"),
        "currency_code" => "SLL",
        "currency_symbol" => "Le",
        "flag_emoji" => "🇸🇱",
      ],
      "SGD" => [
        "name" => $this->stringTranslation->translate("Singapore"),
        "currency_code" => "SGD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇸🇬",
      ],
      "SBD" => [
        "name" => $this->stringTranslation->translate("Solomon Islands"),
        "currency_code" => "SBD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇸🇧",
      ],
      "SOS" => [
        "name" => $this->stringTranslation->translate("Somalia"),
        "currency_code" => "SOS",
        "currency_symbol" => "Sh",
        "flag_emoji" => "🇸🇴",
      ],
      "ZAR" => [
        "name" => $this->stringTranslation->translate("South Africa"),
        "currency_code" => "ZAR",
        "currency_symbol" => "R",
        "flag_emoji" => "🇿🇦",
      ],
      "KRW" => [
        "name" => $this->stringTranslation->translate("South Korea"),
        "currency_code" => "KRW",
        "currency_symbol" => "₩",
        "flag_emoji" => "🇰🇷",
      ],
      "SSP" => [
        "name" => $this->stringTranslation->translate("South Sudan"),
        "currency_code" => "SSP",
        "currency_symbol" => "£",
        "flag_emoji" => "🇸🇸",
      ],
      "LKR" => [
        "name" => $this->stringTranslation->translate("Sri Lanka"),
        "currency_code" => "LKR",
        "currency_symbol" => "₨",
        "flag_emoji" => "🇱🇰",
      ],
      "SDG" => [
        "name" => $this->stringTranslation->translate("Sudan"),
        "currency_code" => "SDG",
        "currency_symbol" => "£",
        "flag_emoji" => "🇸🇩",
      ],
      "SRD" => [
        "name" => $this->stringTranslation->translate("Suriname"),
        "currency_code" => "SRD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇸🇷",
      ],
      "SEK" => [
        "name" => $this->stringTranslation->translate("Sweden"),
        "currency_code" => "SEK",
        "currency_symbol" => "kr",
        "flag_emoji" => "🇸🇪",
      ],
      "SYP" => [
        "name" => $this->stringTranslation->translate("Syria"),
        "currency_code" => "SYP",
        "currency_symbol" => "£",
        "flag_emoji" => "🇸🇾",
      ],
      "TWD" => [
        "name" => $this->stringTranslation->translate("Taiwan"),
        "currency_code" => "TWD",
        "currency_symbol" => "$",
        "flag_emoji" => "🇹🇼",
      ],
      "TJS" => [
        "name" => $this->stringTranslation->translate("Tajikistan"),
        "currency_code" => "TJS",
        "currency_symbol" => "ЅМ",
        "flag_emoji" => "🇹🇯",
      ],
      "TZS" => [
        "name" => $this->stringTranslation->translate("Tanzania"),
        "currency_code" => "TZS",
        "currency_symbol" => "Sh",
        "flag_emoji" => "🇹🇿",
      ],
      "THB" => [
        "name" => $this->stringTranslation->translate("Thailand"),
        "currency_code" => "THB",
        "currency_symbol" => "฿",
        "flag_emoji" => "🇹🇭",
      ],
      "TOP" => [
        "name" => $this->stringTranslation->translate("Tonga"),
        "currency_code" => "TOP",
        "currency_symbol" => "T$",
        "flag_emoji" => "🇹🇴",
      ],
      "TTD" => [
        "name" => $this->stringTranslation->translate("Trinidad and Tobago"),
        "currency_code" => "TTD",
        "currency_symbol" => "TT$",
        "flag_emoji" => "🇹🇹",
      ],
      "TND" => [
        "name" => $this->stringTranslation->translate("Tunisia"),
        "currency_code" => "TND",
        "currency_symbol" => "د.ت",
        "flag_emoji" => "🇹🇳",
      ],
      "TRY" => [
        "name" => $this->stringTranslation->translate("Turkey"),
        "currency_code" => "TRY",
        "currency_symbol" => "₺",
        "flag_emoji" => "🇹🇷",
      ],
      "TMT" => [
        "name" => $this->stringTranslation->translate("Turkmenistan"),
        "currency_code" => "TMT",
        "currency_symbol" => "m",
        "flag_emoji" => "🇹🇲",
      ],
      "UGX" => [
        "name" => $this->stringTranslation->translate("Uganda"),
        "currency_code" => "UGX",
        "currency_symbol" => "Sh",
        "flag_emoji" => "🇺🇬",
      ],
      "UAH" => [
        "name" => $this->stringTranslation->translate("Ukraine"),
        "currency_code" => "UAH",
        "currency_symbol" => "₴",
        "flag_emoji" => "🇺🇦",
      ],
      "AED" => [
        "name" => $this->stringTranslation->translate("United Arab Emirates"),
        "currency_code" => "AED",
        "currency_symbol" => "د.إ",
        "flag_emoji" => "🇦🇪",
      ],
      "GBP" => [
        "name" => $this->stringTranslation->translate("United Kingdom"),
        "currency_code" => "GBP",
        "currency_symbol" => "£",
        "flag_emoji" => "🇬🇧",
      ],
      "UYU" => [
        "name" => $this->stringTranslation->translate("Uruguay"),
        "currency_code" => "UYU",
        "currency_symbol" => "$",
        "flag_emoji" => "🇺🇾",
      ],
      "UZS" => [
        "name" => $this->stringTranslation->translate("Uzbekistan"),
        "currency_code" => "UZS",
        "currency_symbol" => "so'm",
        "flag_emoji" => "🇺🇿",
      ],
      "VUV" => [
        "name" => $this->stringTranslation->translate("Vanuatu"),
        "currency_code" => "VUV",
        "currency_symbol" => "Vt",
        "flag_emoji" => "🇻🇺",
      ],
      "VED" => [
        "name" => $this->stringTranslation->translate("Venezuela"),
        "currency_code" => "VED",
        "currency_symbol" => "Bs.S",
        "flag_emoji" => "🇻🇪",
      ],
      "VND" => [
        "name" => $this->stringTranslation->translate("Vietnam"),
        "currency_code" => "VND",
        "currency_symbol" => "₫",
        "flag_emoji" => "🇻🇳",
      ],
      "YER" => [
        "name" => $this->stringTranslation->translate("Yemen"),
        "currency_code" => "YER",
        "currency_symbol" => "﷼",
        "flag_emoji" => "🇾🇪",
      ],
      "ZMW" => [
        "name" => $this->stringTranslation->translate("Zambia"),
        "currency_code" => "ZMW",
        "currency_symbol" => "ZK",
        "flag_emoji" => "🇿🇲",
      ],
      "ZWL" => [
        "name" => $this->stringTranslation->translate("Zimbabwe"),
        "currency_code" => "ZWL",
        "currency_symbol" => "$",
        "flag_emoji" => "🇿🇼",
      ],
    ];

    // Allow other modules to alter the countries list.
    $this->moduleHandler->alter('exchangerate_countries', $countries);

    return $countries;
  }

}
