<?php

namespace Drupal\exchangerate\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\exchangerate\Service\ExchangeRateApi;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides a form for currency conversion.
 */
class CurrencyConverterForm extends FormBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The exchange rate API service.
   *
   * @var \Drupal\exchangerate\Service\ExchangeRateApi
   */
  protected $apiService;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new CurrencyConverterForm.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\exchangerate\Service\ExchangeRateApi $api_service
   *   The exchange rate API service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(RendererInterface $renderer, ExchangeRateApi $api_service, LoggerChannelFactoryInterface $logger_factory) {
    $this->renderer = $renderer;
    $this->apiService = $api_service;
    $this->logger = $logger_factory->get('exchangerate');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('exchangerate.api'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'currency_converter_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Add a main wrapper class to the form.
    $form['#attributes']['class'][] = 'currency-converter-wrapper';

    // Autocomplete for the base currency selection.
    $form['base_currency'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base Currency'),
      '#description' => $this->t('Enter the base currency or country from which you want to convert.'),
    // Custom route for autocomplete.
      '#autocomplete_route_name' => 'exchangerate.autocomplete',
      '#required' => TRUE,
      '#prefix' => '<div class="currency-converter-input-wrapper currency-converter-base-currency">',
      '#suffix' => '</div>',
    ];

    // Amount input field.
    $form['amount'] = [
      '#type' => 'number',
      '#title' => $this->t('Amount'),
      '#description' => $this->t('Enter the amount in your base currency to convert into target currencies.'),
      '#default_value' => 1,
      '#required' => TRUE,
    // Allow decimal values.
      '#step' => 'any',
      '#attributes' => [
        'placeholder' => $this->t('Enter amount'),
      ],
      '#prefix' => '<div class="currency-converter-input-wrapper currency-converter-amount">',
      '#suffix' => '</div>',
    ];

    // Autocomplete for target currencies (comma-separated list).
    $form['target_currencies'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Target Currencies'),
      '#description' => $this->t('Enter one or more target currencies or countries (comma-separated).'),
    // Custom route for multiple target currencies.
      '#autocomplete_route_name' => 'exchangerate.autocomplete_multiple',
      '#required' => TRUE,
      '#prefix' => '<div class="currency-converter-input-wrapper currency-converter-target">',
      '#suffix' => '</div>',
    ];

    // Results section MUST be placed before the submit button
    // for proper AJAX wrapper.
    $form['conversion_results'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'currency-conversion-results'],
      '#markup' => '',
    ];

    // AJAX submit button.
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Convert'),
      '#ajax' => [
        'callback' => '::convertCurrency',
        'wrapper' => 'currency-conversion-results',
        'method' => 'replace',
        'effect' => 'fade',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Converting currencies...'),
        ],
        'event' => 'click',
      ],
      '#prefix' => '<div class="currency-converter-input-wrapper currency-converter-submit">',
      '#suffix' => '</div>',
    ];

    $form['#cache'] = ['max-age' => -1];

    return $form;
  }

  /**
   * AJAX callback to perform currency conversion and display results.
   */
  public function convertCurrency(array &$form, FormStateInterface $form_state) {
    // Create AJAX response.
    $response = new AjaxResponse();

    // Get form values.
    $base_currency = trim($form_state->getValue('base_currency'));
    $target_currencies_string = $form_state->getValue('target_currencies');
    $amount = $form_state->getValue('amount');

    // Validate that the required fields have values.
    if (empty($base_currency) || empty($target_currencies_string) || empty($amount) || $amount <= 0) {
      $build = [
        '#theme' => 'currency_conversion_results',
        '#results' => [],
        '#errors' => ['Please fill out all fields properly with valid values.'],
        '#amount' => $amount,
        '#base_currency' => $base_currency,
        '#base_currency_flag' => '',
        '#base_currency_symbol' => '',
        '#has_results' => FALSE,
        '#has_errors' => TRUE,
      ];

      $output = $this->renderer->render($build);
      $response->addCommand(new HtmlCommand('#currency-conversion-results', $output));
      return $response;
    }

    // Parse target currencies.
    $target_currencies = array_map('trim', explode(',', $target_currencies_string));
    // Remove empty values.
    $target_currencies = array_filter($target_currencies);

    if (empty($target_currencies)) {
      $build = [
        '#theme' => 'currency_conversion_results',
        '#results' => [],
        '#errors' => ['Please enter at least one target currency.'],
        '#amount' => $amount,
        '#base_currency' => $base_currency,
        '#base_currency_flag' => '',
        '#base_currency_symbol' => '',
        '#has_results' => FALSE,
        '#has_errors' => TRUE,
      ];

      $output = $this->renderer->render($build);
      $response->addCommand(new HtmlCommand('#currency-conversion-results', $output));
      return $response;
    }

    try {
      // Get exchange rates.
      $rates = $this->apiService->getRates();
      $countries = $this->apiService->getCountries();

      $base_currency_symbol = $countries[$base_currency]['currency_symbol'] ?? '';

      if (empty($rates)) {
        $build = [
          '#theme' => 'currency_conversion_results',
          '#results' => [],
          '#errors' => ['Unable to fetch exchange rates. Please try again later.'],
          '#amount' => $amount,
          '#base_currency' => $base_currency,
          '#base_currency_flag' => '',
          '#base_currency_symbol' => $base_currency_symbol,
          '#has_results' => FALSE,
          '#has_errors' => TRUE,
        ];

        $output = $this->renderer->render($build);
        $response->addCommand(new HtmlCommand('#currency-conversion-results', $output));
        return $response;
      }

      // Perform currency conversion logic.
      $results = [];
      $errors = [];

      // Normalize currency codes to uppercase.
      $base_currency = strtoupper($base_currency);
      $base_currency_flag_emoji = $countries[$base_currency]['flag_emoji'] ?? '';

      foreach ($target_currencies as $target_currency) {
        $target_currency = strtoupper(trim($target_currency));

        if (empty($target_currency)) {
          continue;
        }

        if (!isset($rates[$base_currency])) {
          $errors[] = "Base currency '{$base_currency}' not found.";
          continue;
        }

        if (!isset($rates[$target_currency])) {
          $errors[] = "Target currency '{$target_currency}' not found.";
          continue;
        }

        $target_currency_flag_emoji = $countries[$target_currency]['flag_emoji'] ?? '';
        $currency_symbol = $countries[$target_currency]['currency_symbol'] ?? '';
        $country_name = $countries[$target_currency]['name'] ?? '';

        if ($base_currency === $target_currency) {
          $converted_amount = number_format($amount, 2);
        }
        else {
          $converted_amount = ($amount / $rates[$base_currency]) * $rates[$target_currency];
          $converted_amount = number_format($converted_amount, 2);
        }

        $results[$target_currency] = [
          'name' => $country_name,
          'amount' => $converted_amount,
          'flag_emoji' => $target_currency_flag_emoji,
          'currency_symbol' => $currency_symbol,
        ];
      }

      // Build the render array using the theme.
      $build = [
        '#theme' => 'currency_conversion_results',
        '#results' => $results,
        '#errors' => $errors,
        '#amount' => $amount,
        '#base_currency' => $base_currency,
        '#base_currency_flag' => $base_currency_flag_emoji,
        '#base_currency_symbol' => $base_currency_symbol,
        '#has_results' => !empty($results),
        '#has_errors' => !empty($errors),
      ];

    }
    catch (\Exception $e) {
      // Log the error.
      $this->logger->error('Currency conversion error: @message', ['@message' => $e->getMessage()]);

      $build = [
        '#theme' => 'currency_conversion_results',
        '#results' => [],
        '#errors' => ['An error occurred during conversion. Please try again.'],
        '#amount' => $amount,
        '#base_currency' => $base_currency,
        '#base_currency_flag' => '',
        '#base_currency_symbol' => '',
        '#has_results' => FALSE,
        '#has_errors' => TRUE,
      ];
    }

    $output = $this->renderer->render($build);
    $response->addCommand(new HtmlCommand('#currency-conversion-results', $output));
    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This method handles non-AJAX submissions
    // You can add fallback logic here if needed.
  }

}
