<?php

namespace Drupal\exchangerate\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\exchangerate\Service\ExchangeRateApi;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Form\FormBuilderInterface;

/**
 * Provides a 'Currency Converter' block.
 *
 * @Block(
 *   id = "currency_converter_block",
 *   admin_label = @Translation("Currency Converter"),
 *   category = @Translation("Exchange Rate")
 * )
 */
class CurrencyConverterBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The exchange rate API service.
   *
   * @var \Drupal\exchangerate\Service\ExchangeRateApi
   */
  protected $apiService;

  /**
   * The cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * Constructs a new CurrencyConverterBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\exchangerate\Service\ExchangeRateApi $api_service
   *   The exchange rate API service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ExchangeRateApi $api_service, CacheBackendInterface $cache, FormBuilderInterface $form_builder) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiService = $api_service;
    $this->cache = $cache;
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('exchangerate.api'),
      $container->get('cache.default'),
      $container->get('form_builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Build the form.
    $form = $this->formBuilder->getForm('Drupal\exchangerate\Form\CurrencyConverterForm');

    // Return the form directly with proper libraries attached.
    $build = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['currency-converter-block', 'my-custom-block'],
      ],
      'form' => $form,
      '#attached' => [
        'library' => [
      // Essential for AJAX functionality.
          'core/drupal.ajax',
          'exchangerate/currency_converter',
        ],
      ],
      // Disable caching for this block to ensure AJAX works properly.
      '#cache' => [
        'max-age' => 600,
      ],
    ];

    $build['#attached']['library'][] = 'exchangerate/exchangerate_converter';

    return $build;
  }

}
