<?php

namespace Drupal\exchangerate\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\exchangerate\Service\ExchangeRateApi;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Provides an 'Exchange Rate' block.
 *
 * @Block(
 *   id = "exchangerate_block",
 *   admin_label = @Translation("Exchange Rate"),
 *   category = @Translation("Exchange Rate")
 * )
 */
class ExchangeRateBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The exchange rate API service.
   *
   * @var \Drupal\exchangerate\Service\ExchangeRateApi
   */
  protected $apiService;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Constructs a new ExchangeRateBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\exchangerate\Service\ExchangeRateApi $api_service
   *   The exchange rate API service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   The file usage service.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ExchangeRateApi $api_service, EntityTypeManagerInterface $entity_type_manager, FileUsageInterface $file_usage, EntityRepositoryInterface $entity_repository, CacheBackendInterface $cache) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiService = $api_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUsage = $file_usage;
    $this->entityRepository = $entity_repository;
    $this->cache = $cache;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('exchangerate.api'),
      $container->get('entity_type.manager'),
      $container->get('file.usage'),
      $container->get('entity.repository'),
      $container->get('cache.default')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Fetch rates.
    $rates = $this->apiService->getRates();

    $countries = $this->apiService->getCountries();

    $baseCurrency = $this->apiService->getBaseCurrency();
    $baseCurrency = $countries[$baseCurrency];

    // Get selected countries with their order.
    $selectedCountries = $this->configuration['countries'] ?? [];
    $show_base_currency = $this->configuration['show_base_currency'];
    $base_currency_text = $this->configuration['base_currency_text'];
    $description = $this->configuration['description'];
    $layout = $this->configuration['layout'];

    if (isset($description['format']) && $description['format']) {
      $description_text = check_markup($description['value'], $description['format']);
    }
    else {
      $description_text = $description['value'];
    }

    $extra_data = [
      'show_base_currency' => $show_base_currency,
      'base_currency_text' => $base_currency_text,
      'description_text' => $description_text,
      'layout' => $layout,
    ];

    // Filter and order countries based on selection and order.
    $filteredCountries = [];
    if (!empty($selectedCountries)) {
      uasort($selectedCountries, function ($a, $b) {
        return ($a['weight'] ?? 0) <=> ($b['weight'] ?? 0);
      });

      foreach ($selectedCountries as $country_code => $config) {
        if (!empty($config['selected']) && isset($countries[$country_code])) {
          $filteredCountries[$country_code] = $countries[$country_code];
        }
      }
    }

    return [
      '#theme' => 'exchangerate_block',
      '#rates' => $rates,
      '#countries' => $filteredCountries,
      '#base_currency' => $baseCurrency,
      '#extra_data' => $extra_data,
      '#cache' => [
        'keys' => ['exchangerate', 'block_data'],
        'tags' => ['config:exchangerate.settings'],
        'max-age' => 0,
      ],
      '#attached' => [
        'library' => [
          'exchangerate/exchangerate_styles',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    // Get the countries from the service.
    $countries = $this->apiService->getCountries();
    $saved_countries = $this->configuration['countries'] ?? [];

    // Add the "Show Base Currency" checkbox.
    $form['show_base_currency'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Base Currency'),
      '#default_value' => $this->configuration['show_base_currency'] ?? FALSE,
    ];

    // Add the text field for the base currency, only visible if the
    // checkbox is checked.
    $form['base_currency_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base Currency Text'),
      '#default_value' => $this->configuration['base_currency_text'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="settings[show_base_currency]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Text field for adding sidebar text (rich text editor).
    $form['description'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Description'),
      '#default_value' => $this->configuration['description']['value'] ?? '',
      '#allowed_formats' => $this->exchangeRateGetTextFormats(),
      '#format' => $this->configuration['description']['format'] ?? 'basic_html',
      '#description' => $this->t('Enter the text to be displayed at the top of the block content.'),
    ];

    $form['layout'] = [
      '#type' => 'select',
      '#title' => $this->t('Layout'),
      '#default_value' => $this->configuration['layout'] ?? '',
      '#options' => [
        'layout-1' => $this->t('Layout 1'),
        'layout-2' => $this->t('Layout 2'),
        'layout-3' => $this->t('Layout 3'),
        'layout-4' => $this->t('Layout 4'),
      ],
      '#description' => $this->t('Enter the layout of currencies'),
    ];

    // Prepare the table header.
    $header = [
      $this->t('Country'),
      $this->t('Selected'),
      $this->t('Weight'),
    ];

    // Create a sortable table.
    $form['countries'] = [
      '#type' => 'table',
      '#header' => $header,
      '#empty' => $this->t('No countries available.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'country-weight',
        ],
      ],
      '#title' => $this->t('Countries to Display'),
      '#description' => $this->t('Select and order the countries whose exchange rates you want to display. Drag rows to reorder them.'),
    ];

    // Sort countries: selected first (by weight), then unselected
    // alphabetically.
    $selected_countries = [];
    $unselected_countries = [];

    foreach ($countries as $currency_code => $country) {
      if (isset($saved_countries[$currency_code]) && !empty($saved_countries[$currency_code]['selected'])) {
        $selected_countries[$currency_code] = [
          'country' => $country,
          'weight' => $saved_countries[$currency_code]['weight'] ?? 0,
          'selected' => TRUE,
        ];
      }
      else {
        $unselected_countries[$currency_code] = [
          'country' => $country,
        // High weight for unselected.
          'weight' => 999,
          'selected' => FALSE,
        ];
      }
    }

    // Sort selected countries by weight.
    uasort($selected_countries, function ($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });

    // Sort unselected countries alphabetically.
    uasort($unselected_countries, function ($a, $b) {
      return strcasecmp($a['country']['name'], $b['country']['name']);
    });

    // Combine arrays.
    $ordered_countries = $selected_countries + $unselected_countries;

    // Build the form rows.
    $weight = 0;
    foreach ($ordered_countries as $currency_code => $data) {
      $country = $data['country'];
      $flag_emoji = $country['flag_emoji'];

      $form['countries'][$currency_code]['#attributes']['class'][] = 'draggable';
      $form['countries'][$currency_code]['#weight'] = $weight;

      // Country name (not editable)
      $form['countries'][$currency_code]['country'] = [
        '#type' => 'item',
        '#title' => $this->t('@flag @country (@currency)', [
          '@flag' => $flag_emoji,
          '@country' => $country['name'],
          '@currency' => $currency_code,
        ]),
      ];

      // Checkbox for selection.
      $form['countries'][$currency_code]['selected'] = [
        '#type' => 'checkbox',
        '#default_value' => $data['selected'],
        '#title' => $this->t('Select @country', ['@country' => $country['name']]),
        '#title_display' => 'invisible',
      ];

      // Weight field for ordering.
      $form['countries'][$currency_code]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @country', ['@country' => $country['name']]),
        '#title_display' => 'invisible',
        '#default_value' => $data['selected'] ? $data['weight'] : $weight,
        '#attributes' => ['class' => ['country-weight']],
        '#delta' => 100,
      ];

      $weight++;
    }

    $old_description = $this->configuration['description']['value'] ?? '';
    $old_uuids = _editor_parse_file_uuids($old_description);

    $form['old_uuids'] = [
      '#type' => 'value',
      '#value' => $old_uuids,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    // Save selected countries with their weights.
    $countries_config = [];
    $countries_values = $form_state->getValue('countries');

    if ($countries_values) {
      foreach ($countries_values as $country_code => $values) {
        $countries_config[$country_code] = [
          'selected' => !empty($values['selected']),
          'weight' => (int) $values['weight'],
        ];
      }
    }

    // Save other settings.
    $this->configuration['show_base_currency'] = $form_state->getValue('show_base_currency');
    $this->configuration['base_currency_text'] = $form_state->getValue('base_currency_text');
    $this->configuration['countries'] = $countries_config;
    $this->configuration['description'] = $form_state->getValue('description');
    $this->configuration['layout'] = $form_state->getValue('layout');

    // Call the custom submit handler.
    $this->exchangeRateFormSystemThemeSettingsSubmit($form, $form_state);
  }

  /**
   * Submit handler for the theme settings form.
   */
  public function exchangeRateFormSystemThemeSettingsSubmit($form, &$form_state) {
    // Get the submitted rich text value.
    $content = $form_state->getValue('description');
    $description = $content['value'] ?? '';

    if ($description) {
      // Parse the new UUIDs from the submitted content.
      $new_uuids = _editor_parse_file_uuids($description);

      if ($new_uuids) {
        // Process new file UUIDs.
        foreach ($new_uuids as $uuid) {
          if ($file = $this->entityRepository->loadEntityByUuid('file', $uuid)) {
            /** @var \Drupal\file\FileInterface $file */
            if ($file->isTemporary()) {
              $file->setPermanent();
              $file->save();
            }
            $this->fileUsage->add($file, 'exchangerate', 'file', $file->id());
          }
        }
      }
    }

    // Handle removing files that are no longer in the content.
    $old_uuids = $form_state->getValue('old_uuids') ?? [];
    if (!empty($old_uuids)) {
      $uuids_to_remove = array_diff($old_uuids, $new_uuids);

      foreach ($uuids_to_remove as $uuid) {
        if ($file = $this->entityRepository->loadEntityByUuid('file', $uuid)) {
          /** @var \Drupal\file\FileInterface $file */
          if ($file->isPermanent()) {
            $file->setTemporary();
            $file->save();
          }
          $this->fileUsage->delete($file, 'exchangerate', 'file', $file->id());
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'countries' => [],
      'layout' => 'layout-1',
      'description' => [
        'value' => '',
        'format' => 'basic_html',
      ],
    ] + parent::defaultConfiguration();
  }

  /**
   * Get all text format machine names programmatically.
   */
  public function exchangeRateGetTextFormats() {
    // Load all text formats.
    $formats = $this->entityTypeManager->getStorage('filter_format')->loadMultiple();

    // Initialize an array to store the machine names.
    $machine_names = [];

    // Loop through each format and get the machine name.
    foreach ($formats as $format) {
      // Add the machine name to the array.
      $machine_names[] = $format->id();
    }

    // Return the machine names.
    return $machine_names;
  }

}
