<?php

namespace Drupal\exif_manipulate\Hooks;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Error;
use Drupal\exif_manipulate\Services\FileExifProcessorInterface;
use Drupal\file\Entity\File;
use lsolesen\pel\PelException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Hook implementations dealing with file entities.
 */
final class FileEntity implements ContainerInjectionInterface {

  use StringTranslationTrait;

  public function __construct(
    readonly protected FileExifProcessorInterface $exifProcessor,
    readonly protected MessengerInterface $messenger,
    readonly protected LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self(
      $container->get('exif_manipulate.file_exif_processor'),
      $container->get('messenger'),
      $container->get('logger.channel.exif_manipulate'),
    );

    $instance->stringTranslation = $container->get('string_translation');

    return $instance;
  }

  /**
   * Implements hook_ENTITY_TYPE_insert().
   */
  public function fileInsert(File $file): void {
    try {
      $this->exifProcessor->manipulate($file);
    }
    catch (PelException $e) {
      Error::logException(
        $this->logger,
        $e,
        'There was a problem removing the EXIF metadata from %image. %type: @message in %function (line %line of %file).',
        ['%image' => $file->getFileUri()]
      );
      $this->messenger->addWarning('There was a problem removing the metadata from your file.');
    }
  }

}
