<?php

declare(strict_types=1);

namespace Drupal\Tests\exif_manipulate\Functional;

use ColinODell\PsrTestLogger\TestLogger;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Site\Settings;
use Drupal\file\FileInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\UiHelperTrait;
use Psr\Log\LoggerInterface;

/**
 * Tests for the cleanup queue.
 *
 * @group exif_manipulate
 */
class QueueTest extends BrowserTestBase {

  use ImageAssertionTrait;
  use UiHelperTrait;

  const QUEUE_NAME = 'exif_manipulate_clean_exif_data';

  /**
   * The profile to install as a basis for testing.
   *
   * @var string
   */
  protected $profile = 'minimal';

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Admin user account.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $adminUser;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['system', 'image', 'exif_manipulate'];

  /**
   * The file directory for the tested site.
   *
   * @var string
   */
  protected string $fileDir;

  /**
   * The test logger.
   */
  protected LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->adminUser = $this->drupalCreateUser([
      'administer exif manipulate',
    ]);
    $this->drupalLogin($this->adminUser);

    // Test if directories specified in settings exist in filesystem.
    $this->fileDir = Settings::get('file_public_path');
    \Drupal::service('file_system')
      ->prepareDirectory($this->fileDir, FileSystemInterface::CREATE_DIRECTORY);

    $this->logger = new TestLogger();

    $container = \Drupal::getContainer();
    $container->set('logger.channel.exif_manipulate', $this->logger);
  }

  /**
   * Tests that the cleanup queue is working.
   */
  public function testCleanupQueue(): void {
    $file = $this->createEntityForFile('location.jpg');

    $this->drupalGet('admin/config/media/exif_manipulate');
    $this->assertSession()->statusCodeEquals(200);

    // Submit the form to clean the EXIF data.
    $this->submitForm(['directory' => $this->fileDir], 'Clean images');

    // CHeck that we see the success message.
    $this->assertSession()->pageTextContains('The images have been queued for cleaning.');

    // Process the queue.
    $queue = \Drupal::queue(self::QUEUE_NAME);
    $this->processQueue($queue);

    // Check that the queue is now empty.
    $this->assertEquals(0, $queue->numberOfItems(), 'The queue should be empty.');

    // Check that the image was cleaned.
    $this->assertImageIsClean($file);
  }

  /**
   * Test that an invalid image does not trip up the queue.
   */
  public function testQueueWithInvalidImage(): void {
    $file = $this->createEntityForFile('invalid.jpg');

    $this->drupalGet('admin/config/media/exif_manipulate');
    $this->assertSession()->statusCodeEquals(200);

    // Submit the form to clean the EXIF data.
    $this->submitForm(['directory' => $this->fileDir], 'Clean images');

    // Process the queue.
    $queue = \Drupal::queue(self::QUEUE_NAME);
    $this->processQueue($queue);

    // Check that the queue is now empty.
    $this->assertEquals(0, $queue->numberOfItems(), 'The queue should be empty.');
    $this->assertTrue($this->logger->hasErrorThatContains('There was a problem removing the EXIF metadata from %image.'), 'The error message should be logged.');
  }

  /**
   * Creates a file entity for a given file in the tests/files directory.
   */
  public function createEntityForFile(string $filename): FileInterface {
    $image = \Drupal::service('extension.list.module')
      ->getPath('exif_manipulate') . '/tests/files/' . $filename;
    $destination = Settings::get('file_public_path') . '/' . $filename;
    copy($image, $destination);
    $fileStorage = \Drupal::entityTypeManager()->getStorage('file');
    $file = $fileStorage->create([
      'uri' => 'public://' . $filename,
    ]);
    assert($file instanceof FileInterface);
    $file->save();

    return $file;
  }

  /**
   * Processes the queue.
   *
   * @param \Drupal\Core\Queue\QueueInterface $queue
   *   The queue to process.
   */
  protected function processQueue($queue): void {
    // Get the queue worker.
    $queueWorker = \Drupal::service('plugin.manager.queue_worker')
      ->createInstance(self::QUEUE_NAME);

    while ($item = $queue->claimItem()) {
      $queueWorker->processItem($item->data);
      $queue->deleteItem($item);
    }
  }

}
