<?php

declare(strict_types=1);

namespace Drupal\Tests\exif_manipulate\Functional;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Site\Settings;
use Drupal\file\Entity\File;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\UiHelperTrait;

/**
 * Tests stripping Exif data.
 *
 * @group exif_manipulate
 */
class StripTest extends BrowserTestBase {

  use ImageAssertionTrait;
  use UiHelperTrait;

  /**
   * The profile to install as a basis for testing.
   *
   * Using the standard profile to test user picture config provided by the
   * standard profile.
   *
   * @var string
   */
  protected $profile = 'standard';

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Admin user account.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $adminUser;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['system', 'image', 'exif_manipulate'];

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
    ]);
    $this->drupalLogin($this->adminUser);

    // Test if directories specified in settings exist in filesystem.
    $file_dir = Settings::get('file_public_path');
    \Drupal::service('file_system')->prepareDirectory($file_dir, FileSystemInterface::CREATE_DIRECTORY);

    $picture_dir = \Drupal::state()->get('user_picture_path', 'pictures');
    $picture_path = $file_dir . $picture_dir;

    \Drupal::service('file_system')->prepareDirectory($picture_path, FileSystemInterface::CREATE_DIRECTORY);
    $directory_writable = is_writable($picture_path);
    $this->assertTrue($directory_writable, "The directory $picture_path should exist and should be writable.");
  }

  /**
   * Test stripping data from user profile pictures.
   */
  public function testUserPicture() {
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->statusCodeEquals(200);
    $image = \Drupal::service('extension.list.module')->getPath('exif_manipulate') . '/tests/files/location.jpg';
    $file = $this->saveUserPicture($image);
    $this->assertImageIsClean($file);
  }

  /**
   * Test stripping data from an invalid image.
   */
  public function testInvalidImage() {
    $this->drupalLogin($this->adminUser);
    $this->assertSession()->statusCodeEquals(200);
    $image = \Drupal::service('extension.list.module')
      ->getPath('exif_manipulate') . '/tests/files/invalid.jpg';
    $this->saveUserPicture($image);
    $textContent = $this->getTextContent();
    $this->assertStringContainsString('There was a problem removing the metadata from your file.', $textContent, 'Feedback must be give about the failed operation.');
  }

  /**
   * Uploads a user picture.
   */
  private function saveUserPicture($image) {
    $edit = ['files[user_picture_0]' => \Drupal::service('file_system')->realpath($image)];
    $uid = $this->adminUser->id();
    $this->drupalGet('user/' . $uid . '/edit');
    $this->submitForm($edit, 'Save');

    // Load actual user data from database.
    $user_storage = $this->container->get('entity_type.manager')->getStorage('user');
    $user_storage->resetCache([$this->adminUser->id()]);

    $account = $user_storage->load($this->adminUser->id());
    return File::load($account->user_picture->target_id);
  }

}
